import fs from "fs";
import path from "path";
import { fileURLToPath } from "url";
import { dirname } from "path";
const __filename = fileURLToPath(import.meta.url);
const __dirname = dirname(__filename);
/** Function to dynamically create and update controller files */
export const createControllerFile = (controllerName) => {
  const controllerDir = path.join(__dirname, "..", "controller");
  const filePath = path.join(controllerDir, `${controllerName}Controller.js`);

  // Check if the file already exists
  if (fs.existsSync(filePath)) {
    console.log(
      `Controller file ${filePath} already exists. Skipping creation.`
    );
    return;
  }

  // Define the content of the controller file
  const fileContent = `
  import db from "../db-config.js";
  import { sendResponse } from "../helper/wrapper.js";
  import { 
    insertActivityLog, 
    searchConditionRecord, 
    makeJoins, 
    whereCondition, 
    countQueryCondition, 
    settingsUpdater,
    deleteSettingRecord, 
    encodeSingle_statement, 
    decodeSingle_statement ,
  } from "../helper/general.js";
  import ${capitalize(controllerName)} from "../sequelize/${capitalize(
    controllerName
  )}Schema.js";

  /** Function to create or update ${capitalize(controllerName)} */
  export const createUpdate${capitalize(controllerName)} = async (req, res) => {
    const { id, name, description } = req.body;
    const encodedDescription = await encodeSingle_statement(description);

    if (id) {
      const result = await settingsUpdater(
        ${capitalize(controllerName)},
        req.body.organization,
        req.body,
        req.user
      );
      if (!result) {
        return sendResponse(res, 404, "No data found");
      }
      await insertActivityLog(req.user.sessionid, "update", "${capitalize(
        controllerName
      )}", \`This user updated ${capitalize(controllerName)} with id: \${id}\`);
      return sendResponse(res, 200, "Record updated successfully");
    } else {
      const insertPromises = req.body.organization.map(async (org) => {
        const [checkName] = await db.query(\`SELECT name FROM ${tableNameMaker(
          controllerName
        )} WHERE name = ? AND deleted = 0 AND organization = ?\`, [name, org]);
        if (checkName.length === 0) {
          await db.query(\`INSERT INTO ${tableNameMaker(
            controllerName
          )} (name, description, created_by, organization) VALUES (?, ?, ?, ?)\`, [name, encodedDescription, req.user.sessionid, org]);
          await insertActivityLog(req.user.sessionid, "create", "${capitalize(
            controllerName
          )}", \`This user created a new ${capitalize(
    controllerName
  )} '\${name}' for organization \${org}\`);
        }
      });
      await Promise.all(insertPromises);
      return sendResponse(res, 200, "Record created successfully");
    }
  };

  /** Function to view all or single ${capitalize(controllerName)} */
  export const view${capitalize(controllerName)} = async (req, res) => {
    const { id } = req.params;
    const condition = await whereCondition({
      table: "${tableNameMaker(controllerName)}",
      page: req.query.page,
      all: req.query.all,
      pageSize: req.query.pageSize,
      filter: req.query.filter,
      id,
      user: req.user,
      grouped: req.query.grouped,
    });

    const searchFields = [
      "${tableNameMaker(controllerName)}.name",
      "${tableNameMaker(controllerName)}.description",
      "users.name",
      "organization.name"
    ];
    let searchCondition = await searchConditionRecord(req.query.search, searchFields);
    const joins = [
      { type: "left", targetTable: "users", onCondition: "users.id = ${tableNameMaker(
        controllerName
      )}.created_by" },
      { type: "left", targetTable: "organization", onCondition: "organization.id = ${tableNameMaker(
        controllerName
      )}.organization" }
    ];
    const joinsRecord = await makeJoins(joins);

    const fetchQuery = \`SELECT ${tableNameMaker(
      controllerName
    )}.*, CONCAT(users.name, ' ', users.surname) AS created_by, organization.name AS organization_name FROM ${tableNameMaker(
    controllerName
  )} \${joinsRecord} WHERE ${tableNameMaker(
    controllerName
  )}.deleted = 0 \${searchCondition} \${condition}\`;
    let [records] = await db.query(fetchQuery);

    for (const record of records) {
      record.description = await decodeSingle_statement(record.description);
       if (req.query.grouped == "true") {
        const [organizations] = await db.query(
          \`SELECT organization FROM ${tableNameMaker(
            controllerName
          )} WHERE deleted = 0 AND name = ?\`,
          [record.name]
        );
        const arr = organizations.map((item) => item.organization);
        record.organizations = arr;
      }
    }

    const totalRecord = await countQueryCondition(fetchQuery);
    return sendResponse(res, 200, records, totalRecord);
  };

  /** Function to delete ${capitalize(controllerName)} */
  export const delete${capitalize(controllerName)} = async (req, res) => {
    const { id } = req.params;
    const deleteRecord = await deleteSettingRecord("${tableNameMaker(
      controllerName
    )}", id);
    if (deleteRecord) {
      await insertActivityLog(req.user.sessionid, "delete", "${capitalize(
        controllerName
      )}", id);
      return sendResponse(res, 200, "Record deleted successfully");
    } else {
      return sendResponse(res, 404, "Record not found");
    }
  };
  `;

  // Create the controllers directory if it doesn't exist
  if (!fs.existsSync(controllerDir)) {
    fs.mkdirSync(controllerDir, { recursive: true });
  }

  // Write the content to the file
  fs.writeFileSync(filePath, fileContent.trim());
  console.log(`Controller file created at ${filePath}`);
};

// Utility function to create routes file
export const createRoutesFile = (entityName) => {
  const routesDir = path.join(__dirname, "..", "routes");
  const filePath = path.join(routesDir, `${entityName}Routes.js`);

  // Check if the file already exists
  if (fs.existsSync(filePath)) {
    console.log(`Routes file ${filePath} already exists. Skipping creation.`);
    return;
  }
  const capitalizedName = capitalize(entityName);
  const routeName = routeNameMaker(entityName);
  const tableName = tableNameMaker(entityName);
  const headingName = symbolAdder(capitalizedName);
  // Define the content of the routes file
  const fileContent = `
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import { createUpdate${capitalizedName}, delete${capitalizedName}, view${capitalizedName} } from "../controller/${entityName}Controller.js";

/**
 * @swagger
 * tags:
 *   name: ${headingName}
 *   description: API operations for managing ${headingName}
 */
const ${entityName}Routes = Router();

/**
 * @swagger
 * /api/${entityName}/create-update-${routeName}:
 *   post:
 *     summary: Create or update a ${headingName}
 *     tags: [${headingName}]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example: { id: "id", title: "title", description: "description" }
 */
${entityName}Routes.post(
  "/create-update-${routeName}",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(req.body.id ? "edit" : "create", 405 , "${tableName}")(req, res, next);
  },
  controllerWrapper(createUpdate${capitalizedName})
);

/**
 * @swagger
 * /api/${entityName}/get-${routeName}:
 *   get:
 *     summary: Get all records of ${headingName}
 *     tags: [${headingName}]
 */
${entityName}Routes.get(
  "/get-${routeName}",
  verifyToken,
  sideBarPermissionCheck("view", 405, "${tableName}"),
  controllerWrapper(view${capitalizedName})
);

/**
 * @swagger
 * /api/${entityName}/get-${routeName}/{id}:
 *   get:
 *     summary: Get a specific record of ${headingName}
 *     tags: [${headingName}]
 */
${entityName}Routes.get(
  "/get-${routeName}/:id",
  verifyToken,
  sideBarPermissionCheck("view", 405, "${tableName}"),
  controllerWrapper(view${capitalizedName})
);

/**
 * @swagger
 * /api/${entityName}/delete-${routeName}/{id}:
 *   delete:
 *     summary: Delete a specific record of ${headingName}
 *     tags: [${headingName}]
 */
${entityName}Routes.delete(
  "/delete-${routeName}/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 405, "${tableName}"),
  controllerWrapper(delete${capitalizedName})
);

export default ${entityName}Routes;
`;

  // Create the routes directory if it doesn't exist
  if (!fs.existsSync(routesDir)) {
    fs.mkdirSync(routesDir, { recursive: true });
  }

  // Write the content to the file
  fs.writeFileSync(filePath, fileContent.trim());
  console.log(`Routes file created at ${filePath}`);
};

// Function to capitalize the first letter of a string
const capitalize = (str) => {
  return str.charAt(0).toUpperCase() + str.slice(1);
};

/** Function to generate table name using entity name */
const tableNameMaker = (str) => {
  let result = "";
  for (let i = 0; i < str.length; i++) {
    const char = str.charAt(i);
    if (char === char.toUpperCase()) {
      result += "_" + char.toLowerCase();
    } else {
      result += char;
    }
  }
  if (result.charAt(0) === "_") {
    result = result.slice(1);
  }
  return result;
};

/** Function to generate table name using entity name */
const symbolAdder = (str, symbol = " ") => {
  let result = "";
  for (let i = 0; i < str.length; i++) {
    const char = str.charAt(i);
    if (char === char.toUpperCase()) {
      result += symbol + char.toLowerCase();
    } else {
      result += char;
    }
  }
  if (result.charAt(0) === "_") {
    result = result.slice(1);
  }
  return result;
};

/** Function to generate route name using entity name */
const routeNameMaker = (str) => {
  let result = "";
  for (let i = 0; i < str.length; i++) {
    const char = str.charAt(i);
    if (char === char.toUpperCase()) {
      result += "-" + char.toLowerCase();
    } else {
      result += char;
    }
  }
  if (result.charAt(0) === "_") {
    result = result.slice(1);
  }
  return result;
};

// Function to create Sequelize model file with predefined fields
export const createModelFile = (entityName) => {
  const modelDir = path.join(__dirname, "..", "sequelize"); // Adjust as per your directory structure
  const filePath = path.join(modelDir, `${capitalize(entityName)}Schema.js`);

  // Check if the file already exists
  if (fs.existsSync(filePath)) {
    console.log(`Model file ${filePath} already exists. Skipping creation.`);
    return;
  }

  // Define the content of the model file with predefined fields
  const fileContent = `
import { DataTypes } from "sequelize";
import sequelize from "./sequelize.js"; // Adjust the import path as necessary
import User from "./UserSchema.js"; // Adjust imports according to your project
import Organization from "./OrganizationSchema.js";
import Department from "./DepartmentSchema.js";

// Define the ${capitalize(entityName)} model
const ${capitalize(entityName)} = sequelize.define(
  "${tableNameMaker(entityName)}", 
  {
    id: {
      type: DataTypes.INTEGER,
      autoIncrement: true,
      primaryKey: true,
    },
    name: {
      type: DataTypes.STRING(250),
      allowNull: true,
    },
    description: {
      type: DataTypes.TEXT("long"),
      allowNull: true,
    },
    department: {
      type: DataTypes.INTEGER,
      allowNull: true,
      references: {
        model: Department,
        key: "id",
      },
      //onUpdate: "CASCADE",
      //onDelete: "SET NULL",
    },
    organization: {
      type: DataTypes.INTEGER,
      allowNull: true,
      references: {
        model: Organization,
        key: "id",
      },
      //onUpdate: "CASCADE",
      //onDelete: "SET NULL",
    },
    deleted: {
      type: DataTypes.INTEGER,
      allowNull: true,
      defaultValue: 0,
    },
    created_at: {
      type: DataTypes.DATE,
      allowNull: true,
      defaultValue: sequelize.literal("CURRENT_TIMESTAMP"),
    },
    created_by: {
      type: DataTypes.INTEGER,
      allowNull: true,
      references: {
        model: User,
        key: "id",
      },
      //onUpdate: "CASCADE",
      //onDelete: "SET NULL",
    },
    updated_at: {
      type: DataTypes.DATE,
      allowNull: true,
      defaultValue: sequelize.literal("CURRENT_TIMESTAMP"),
      onUpdate: sequelize.literal("CURRENT_TIMESTAMP"),
    },
    updated_by: {
      type: DataTypes.INTEGER,
      allowNull: true,
      references: {
        model: User,
        key: "id",
      },
      //onUpdate: "CASCADE",
      //onDelete: "SET NULL",
    },
  },
  {
    tableName: "${tableNameMaker(entityName)}",
    timestamps: false,

    charset: "utf8mb4",
    collate: "utf8mb4_general_ci",
  }
);

// Associations
${capitalize(entityName)}.belongsTo(Department, {
  as: "departmentDetails",
  foreignKey: "department",
});
${capitalize(entityName)}.belongsTo(Organization, {
  as: "organizationDetails",
  foreignKey: "organization",
});
${capitalize(entityName)}.belongsTo(User, {
  as: "creator",
  foreignKey: "created_by",
});
${capitalize(entityName)}.belongsTo(User, {
  as: "updater",
  foreignKey: "updated_by",
});
console.log(await ${capitalize(entityName)}.sync({ alter: true }));
export default ${capitalize(entityName)};
`;

  // Create the model directory if it doesn't exist
  if (!fs.existsSync(modelDir)) {
    fs.mkdirSync(modelDir, { recursive: true });
  }

  // Write the content to the file
  fs.writeFileSync(filePath, fileContent.trim());
  console.log(`Model file created at ${filePath}`);
};

// Usage example:

createControllerFile("category");
createRoutesFile("category");
createModelFile("category");
