// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import {
  createUpdateSoftwareAccessRequest,
  deleteSoftwareAccessRequest,
  getPermission,
  updateAccessRequestStatus,
  viewSoftwareAccessRequest,
} from "../controller/softwareAccessRequestController.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import { controllerWrapper } from "../helper/wrapper.js";
import softwareAccessRoutes from "./softwareAccessRoutes.js";

/**
 * @swagger
 * tags:
 *   name: Software Access Request
 *   description: API operations for managing all Software Access Request
 */
const SoftwareAccessRequestRoutes = Router();

/**
 * @swagger
 * /api/SoftwareAccessRequest/create-update-software-access-request:
 *   post:
 *     summary: Create or update an Software Access Request
 *     tags: [Software Access Request]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: string
 *               reason:
 *                 type: string
 *               organization:
 *                 type: number
 *               sidebar_id:
 *                 type: number
 *               user_id:
 *                 type: string
 *               status:
 *                 type: string
 *                 items:
 *                   type: string
 *           example:
 *             id: "id"
 *             reason: "Reason for the software-access-request"
 *             organization: "Organization1"
 *             sidebar_id: "23"
 *             status: "Pending"
 *             user_id: "User ID"
 *     responses:
 *       200:
 *         description: Software Access Request created or updated successfully
 *       400:
 *         description: Bad request. Invalid input parameters
 *       500:
 *         description: Internal server error
 */
SoftwareAccessRequestRoutes.post(
  "/create-update-software-access-request",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(req.body.id ? "edit" : "create", 373, "software_access_request")(req, res, next);
  },
  controllerWrapper(createUpdateSoftwareAccessRequest)
);

/**
 * @swagger
 * /api/SoftwareAccessRequest/get-software-access-request:
 *   get:
 *     summary: Get list of all SoftwareAccessRequest
 *     tags: [Software Access Request]
 *     responses:
 *       200:
 *         description: List of SoftwareAccessRequest
 *       500:
 *         description: Internal server error
 */
SoftwareAccessRequestRoutes.get(
  "/get-software-access-request",
  verifyToken,
  sideBarPermissionCheck("view", 373, "software_access_request"),
  controllerWrapper(viewSoftwareAccessRequest)
);

/**
 * @swagger
 * /api/SoftwareAccessRequest/get-software-access-request/{id}:
 *   get:
 *     summary: Get a specific record of Software Access Request
 *     tags: [Software Access Request]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: string
 *         required: true
 *         description: The ID of the software-access-request
 *     responses:
 *       200:
 *         description: Software Access Request data
 *       404:
 *         description: Software Access Request not found
 *       500:
 *         description: Internal server error
 */
SoftwareAccessRequestRoutes.get(
  "/get-software-access-request/:id",
  verifyToken,
  sideBarPermissionCheck("view", 373, "software_access_request"),
  controllerWrapper(viewSoftwareAccessRequest)
);

/**
 * @swagger
 * /api/SoftwareAccessRequest/delete-software-access-request/{id}:
 *   delete:
 *     summary: Delete a specific record of Software Access Request
 *     tags: [Software Access Request]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: string
 *         required: true
 *         description: The ID of the software-access-request to delete
 *     responses:
 *       200:
 *         description: Software Access Request successfully deleted
 *       404:
 *         description: Software Access Request not found
 *       500:
 *         description: Internal server error
 */
SoftwareAccessRequestRoutes.delete(
  "/delete-software-access-request/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 373, "software_access_request"),
  controllerWrapper(deleteSoftwareAccessRequest)
);

/**
 * @swagger
 * /api/SoftwareAccessRequest/update-request-status:
 *   post:
 *     summary: Approve or reject Software Access Request
 *     tags: [Software Access Request]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: string
 *               status:
 *                 type: string
 *                 items:
 *                   type: string
 *           example:
 *             id: "id"
 *             status: "Pending"
 *     responses:
 *       200:
 *         description: Software Access Request updated successfully
 *       400:
 *         description: Bad request. Invalid input parameters
 *       500:
 *         description: Internal server error
 */
SoftwareAccessRequestRoutes.post("/update-request-status", verifyToken, controllerWrapper(updateAccessRequestStatus));

/**
 * @swagger
 * /api/SoftwareAccessRequest/get-permission:
 *   get:
 *     summary: Get all permission for logged in user
 *     tags: [Software Access Request]
 *     responses:
 *       200:
 *         description: Permission data
 *       500:
 *         description: Internal server error
 */
SoftwareAccessRequestRoutes.get("/get-permission", verifyToken, controllerWrapper(getPermission));

export default SoftwareAccessRequestRoutes;
