// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import {
  createUpdateRole,
  deleteRole,
  viewRole,
  importRoleData,
} from "../controller/rolesController.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import { validateRequest } from "../helper/general.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { rolesCreateUpdateValidationSchema } from "../validationsRecord/validationSchema.js";

/**
 * @swagger
 * tags:
 *   name: Role
 *   description: API operations for managing all Roles
 */
const rolesRoutes = Router();

/**
 * @swagger
 * /api/roles/create-update-role:
 *   post:
 *     summary: Create or Update a Role
 *     tags: [Role]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: string
 *               level:
 *                 type: string
 *               name:
 *                 type: string
 *               description:
 *                 type: string
 *               organization:
 *                 type: string
 *               responsibilities:
 *                 type: string
 *               qualification:
 *                 type: string
 *               license:
 *                 type: string
 *               year_of_experience:
 *                 type: string
 *               skills:
 *                 type: string
 *               hierarchy:
 *                 type: string
 *               other_requirements:
 *                 type: string
 *               select_other_requirements:
 *                 type: string
 *               department:
 *                 type: string
 *           example:
 *             id: "1"
 *             level: "Senior"
 *             name: "Senior Developer"
 *             description: "Responsible for leading projects."
 *             organization: "Tech Corp"
 *             responsibilities: "Development"
 *             qualification: "Bachelor's Degree"
 *             license: "N/A"
 *             year_of_experience: "5"
 *             skills: "JavaScript, Node.js"
 *             hierarchy: "Senior"
 *             other_requirements: "Leadership skills"
 *             select_other_requirements: "N/A"
 *             department: "Engineering"
 */
rolesRoutes.post(
  "/create-update-role",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(req.body.id ? "edit" : "create", 7, "roles")(
      req,
      res,
      next
    );
  },
  validateRequest(rolesCreateUpdateValidationSchema),
  controllerWrapper(createUpdateRole)
);

/**
 * @swagger
 * /api/roles/get-roles:
 *   get:
 *     summary: Get All Roles List
 *     tags: [Role]
 */
rolesRoutes.get(
  "/get-roles",
  verifyToken,
  sideBarPermissionCheck("view", 7, "roles"),
  controllerWrapper(viewRole)
);

/**
 * @swagger
 * /api/roles/get-roles/{id}:
 *   get:
 *     summary: Get A Specific Role Record
 *     tags: [Role]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: string
 *         description: The ID of the Role
 */
rolesRoutes.get(
  "/get-roles/:id",
  verifyToken,
  sideBarPermissionCheck("view", 7, "roles"),
  controllerWrapper(viewRole)
);

/**
 * @swagger
 * /api/roles/delete-roles/{id}:
 *   delete:
 *     summary: Delete A Specific Record Of Roles
 *     tags: [Role]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: string
 *         description: The ID of the Role to delete
 */
rolesRoutes.delete(
  "/delete-roles/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 7, "roles"),
  controllerWrapper(deleteRole)
);

/**
 * @swagger
 * /api/roles/import-roles-data:
 *   post:
 *     summary: Import Record Of Roles
 *     tags: [Role]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               organization:
 *                 type: integer
 *               data:
 *                 type: array
 *                 items:
 *                   type: object
 *                   properties:
 *                     name:
 *                       type: string
 *                     description:
 *                       type: string
 *                     level:
 *                       type: integer
 *                     hierarchy:
 *                       type: string
 *                     unilateral:
 *                       type: string
 *                     qualification:
 *                       type: string
 *                     license:
 *                       type: string
 *                     other:
 *                       type: string
 *                     year_of_experience:
 *                       type: string
 *                     skills:
 *                       type: string
 *                     other_requirements:
 *                       type: string
 *           example:
 *             organization: 1
 *             data:
 *               - name: "Junior Developer"
 *                 description: "Entry-level developer."
 *                 level: 1
 *                 hierarchy: "Junior"
 *                 unilateral: "N/A"
 *                 qualification: "Associate's Degree"
 *                 license: "N/A"
 *                 other: "N/A"
 *                 year_of_experience: "1"
 *                 skills: "JavaScript"
 *                 other_requirements: "N/A"
 */
rolesRoutes.post(
  "/import-roles-data",
  verifyToken,
  sideBarPermissionCheck("create", 7, "roles"),
  controllerWrapper(importRoleData)
);

export default rolesRoutes;
