import { createUpdateRetentionPolicySetting, deleteRetentionPolicySetting, viewRetentionPolicySetting } from "../controller/retentionPolicySettingController.js";
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import { controllerWrapper } from "../helper/wrapper.js";

/**
 * @swagger
 * tags:
 *   name: Retention Policy Setting
 *   description: API operations for managing all RetentionPolicySettings
 */
const retentionPolicySettingRoutes = Router();

/**
 * @swagger
 * /api/retentionPolicySetting/create-update-retention-policy-setting:
 *   post:
 *     summary: Create or Update a RetentionPolicySetting
 *     tags: [Retention Policy Setting]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: string
 *               level:
 *                 type: string
 *               name:
 *                 type: string
 *               description:
 *                 type: string
 *               organization:
 *                 type: string
 *               responsibilities:
 *                 type: string
 *               qualification:
 *                 type: string
 *               license:
 *                 type: string
 *               year_of_experience:
 *                 type: string
 *               skills:
 *                 type: string
 *               hierarchy:
 *                 type: string
 *               other_requirements:
 *                 type: string
 *               select_other_requirements:
 *                 type: string
 *               department:
 *                 type: string
 *           example:
 *             id: "1"
 *             level: "Senior"
 *             name: "Senior Developer"
 *             description: "Responsible for leading projects."
 *             organization: "Tech Corp"
 *             responsibilities: "Development"
 *             qualification: "Bachelor's Degree"
 *             license: "N/A"
 *             year_of_experience: "5"
 *             skills: "JavaScript, Node.js"
 *             hierarchy: "Senior"
 *             other_requirements: "Leadership skills"
 *             select_other_requirements: "N/A"
 *             department: "Engineering"
 */
retentionPolicySettingRoutes.post(
  "/create-update-retention-policy-setting",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(req.body.id ? "edit" : "create", 7, "retention_policy_setting")(
      req,
      res,
      next
    );
  },
  controllerWrapper(createUpdateRetentionPolicySetting)
);

/**
 * @swagger
 * /api/retentionPolicySetting/get-retention-policy-setting:
 *   get:
 *     summary: Get All RetentionPolicySettings List
 *     tags: [Retention Policy Setting]
 */
retentionPolicySettingRoutes.get(
  "/get-retention-policy-setting",
  verifyToken,
  sideBarPermissionCheck("view", 7, "retention_policy_setting"),
  controllerWrapper(viewRetentionPolicySetting)
);

/**
 * @swagger
 * /api/retentionPolicySetting/get-retention-policy-setting/{id}:
 *   get:
 *     summary: Get A Specific RetentionPolicySetting Record
 *     tags: [Retention Policy Setting]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: string
 *         description: The ID of the RetentionPolicySetting
 */
retentionPolicySettingRoutes.get(
  "/get-retention-policy-setting/:id",
  verifyToken,
  sideBarPermissionCheck("view", 7, "retention_policy_setting"),
  controllerWrapper(viewRetentionPolicySetting)
);

/**
 * @swagger
 * /api/retentionPolicySetting/delete-retention-policy-setting/{id}:
 *   delete:
 *     summary: Delete A Specific Record Of RetentionPolicySettings
 *     tags: [Retention Policy Setting]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: string
 *         description: The ID of the RetentionPolicySetting to delete
 */
retentionPolicySettingRoutes.delete(
  "/delete-retention-policy-setting/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 7, "retention_policy_setting"),
  controllerWrapper(deleteRetentionPolicySetting)
);


export default retentionPolicySettingRoutes;