import { Router } from "express";
import {
  createUpdateResponsibility,
  deleteResponsibility,
  viewAllResponsibility,
} from "../controller/responsibilityController.js";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import { validateRequest } from "../helper/general.js";
import { responsibilityCreateUpdateValidationSchema } from "../validationsRecord/validationSchema.js";
import { controllerWrapper } from "../helper/wrapper.js";

/**
 * @swagger
 * tags:
 *   name: Responsibility
 *   description: API operations for managing all Responsibilities
 */
const responsibilityRoutes = Router();

/**
 * @swagger
 * /api/responsibility/create-update-responsibility:
 *   post:
 *     summary: Create or Update Responsibilities
 *     tags: [Responsibility]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: string
 *               name:
 *                 type: string
 *               description:
 *                 type: string
 *               organization:
 *                 type: string
 *               department:
 *                 type: string
 *                 nullable: true
 *           example:
 *             id: "1"
 *             name: "Responsibility Name"
 *             description: "Responsibility Description"
 *             organization: "Organization Name"
 *             department: "Optional Department"
 */
responsibilityRoutes.post(
  "/create-update-responsibility",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      168,
      "responsibility"
    )(req, res, next);
  },
  validateRequest(responsibilityCreateUpdateValidationSchema),
  controllerWrapper(createUpdateResponsibility)
);

/**
 * @swagger
 * /api/responsibility/get-responsibility:
 *   get:
 *     summary: Get All Responsibilities
 *     tags: [Responsibility]
 */
responsibilityRoutes.get(
  "/get-responsibility",
  verifyToken,
  sideBarPermissionCheck("view", 168, "responsibility"),
  controllerWrapper(viewAllResponsibility)
);

/**
 * @swagger
 * /api/responsibility/get-responsibility/{id}:
 *   get:
 *     summary: Get Specific Responsibility
 *     tags: [Responsibility]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: string
 *         description: The ID of the Responsibility
 */
responsibilityRoutes.get(
  "/get-responsibility/:id",
  verifyToken,
  sideBarPermissionCheck("view", 168, "responsibility"),
  controllerWrapper(viewAllResponsibility)
);

/**
 * @swagger
 * /api/responsibility/delete-responsibility/{id}:
 *   delete:
 *     summary: Delete Specific Responsibility
 *     tags: [Responsibility]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: string
 *         description: The ID of the Responsibility
 */
responsibilityRoutes.delete(
  "/delete-responsibility/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 168, "responsibility"),
  controllerWrapper(deleteResponsibility)
);

export default responsibilityRoutes;
