// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import {
  createUpdateMaterialClassification,
  deleteMaterialClassification,
  viewMaterialClassification,
} from "../controller/materialClassificationController.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import { controllerWrapper } from "../helper/wrapper.js";

/**
 * @swagger
 * tags:
 *   name: Material Classification
 *   description: API operations for managing all Material Classification
 */
const materialClassificationRoutes = Router();

/**
 * @swagger
 * /api/materialClassification/create-update-material-classification:
 *   post:
 *     summary: Create or update an Material Classification
 *     tags: [Material Classification]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: string
 *               name:
 *                 type: string
 *               description:
 *                 type: string
 *               organization:
 *                 type: array
 *                 items:
 *                   type: string
 *           example:
 *             id: "id"
 *             name: "Material Classification"
 *             description: "Description of the material classification"
 *             organization: ["Organization1", "Organization2"]
 *     responses:
 *       200:
 *         description: Material Classification created or updated successfully
 *       400:
 *         description: Bad request. Invalid input parameters
 *       500:
 *         description: Internal server error
 */
materialClassificationRoutes.post(
  "/create-update-material-classification",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      326,
      "material_classification"
    )(req, res, next);
  },
  controllerWrapper(createUpdateMaterialClassification)
);

/**
 * @swagger
 * /api/materialClassification/get-material-classification:
 *   get:
 *     summary: Get list of all Material Classifications
 *     tags: [Material Classification]
 *     responses:
 *       200:
 *         description: List of material classifications
 *       500:
 *         description: Internal server error
 */
materialClassificationRoutes.get(
  "/get-material-classification",
  verifyToken,
  sideBarPermissionCheck("view", 326, "material_classification"),
  controllerWrapper(viewMaterialClassification)
);

/**
 * @swagger
 * /api/materialClassification/get-material-classification/{id}:
 *   get:
 *     summary: Get a specific record of Material Classification
 *     tags: [Material Classification]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: string
 *         required: true
 *         description: The ID of the material classification
 *     responses:
 *       200:
 *         description: Material Classification data
 *       404:
 *         description: Material Classification not found
 *       500:
 *         description: Internal server error
 */
materialClassificationRoutes.get(
  "/get-material-classification/:id",
  verifyToken,
  sideBarPermissionCheck("view", 326, "material_classification"),
  controllerWrapper(viewMaterialClassification)
);

/**
 * @swagger
 * /api/materialClassification/delete-material-classification/{id}:
 *   delete:
 *     summary: Delete a specific record of Material Classification
 *     tags: [Material Classification]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: string
 *         required: true
 *         description: The ID of the material classification to delete
 *     responses:
 *       200:
 *         description: Material Classification successfully deleted
 *       404:
 *         description: Material Classification not found
 *       500:
 *         description: Internal server error
 */
materialClassificationRoutes.delete(
  "/delete-material-classification/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 326, "material_classification"),
  controllerWrapper(deleteMaterialClassification)
);

export default materialClassificationRoutes;
