import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import {
  createUpdateIllnessType,
  deleteIllnessType,
  getParentIllnessTypeList,
  viewIllnessType,
  viewIllnessTypeDropdown,
} from "../controller/illnessTypeController.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";

const illnessTypeRoutes = Router();

/**
 * @swagger
 * tags:
 *   name: Illness Type
 *   description: API operations for managing all Template Categories
 */

/**
 * @swagger
 * /api/illnessType/create-update-illness-type:
 *   post:
 *     summary: Create or update an Illness Type
 *     tags: [Illness Type]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: integer
 *               name:
 *                 type: string
 *               description:
 *                 type: string
 *               organization:
 *                 type: array
 *                 items:
 *                   type: integer
 *           example:
 *             id: 1
 *             name: "Safety Incident"
 *             description: "Category for safety-related incidents"
 *             category : "category"
 *             sub_category: "sub category"
 *             organization: [1, 2]
 */
illnessTypeRoutes.post(
  "/create-update-illness-type",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      249,
      "illness_type"
    )(req, res, next);
  },
  controllerWrapper(createUpdateIllnessType)
);

/**
 * @swagger
 * /api/illnessType/get-illness-type:
 *   get:
 *     summary: Get list of all Incident Categories
 *     tags: [Illness Type]
 */
illnessTypeRoutes.get(
  "/get-illness-type",
  verifyToken,
  controllerWrapper(viewIllnessType)
);

/**
 * @swagger
 * /api/illnessType/get-illness-type/{id}:
 *   get:
 *     summary: Get a specific record of an Illness Type
 *     tags: [Illness Type]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: ID of the Illness Type to retrieve
 */
illnessTypeRoutes.get(
  "/get-illness-type/:id",
  verifyToken,
  controllerWrapper(viewIllnessType)
);

/**
 * @swagger
 * /api/illnessType/delete-illness-type/{id}:
 *   delete:
 *     summary: Delete a specific record of an Illness Type
 *     tags: [Illness Type]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: ID of the Illness Type to delete
 */
illnessTypeRoutes.delete(
  "/delete-illness-type/:id",
  verifyToken,
  controllerWrapper(deleteIllnessType)
);

/**
 * @swagger
 * /api/illnessType/get-illness-type-list:
 *   get:
 *     summary: Get the list for selecting parent category
 *     tags: [Illness Type]
 *     parameters:
 *       - in: query
 *         organization : [1,2,34,4]
 *         schema:
 *           type: Array
 *         description: Array of organization to fetch the list
 */

illnessTypeRoutes.get(
  "/get-illness-type-list",
  verifyToken,
  controllerWrapper(getParentIllnessTypeList)
);



/**
 * @swagger
 * /api/illnessType/get-illness-type-dropdown:
 *   get:
 *     summary: Get the list all Illness Type
 *     tags: [Illness Type]
 */

illnessTypeRoutes.get(
  "/get-illness-type-dropdown",
  verifyToken,
  controllerWrapper(viewIllnessTypeDropdown)
);





export default illnessTypeRoutes;
