import { Router } from "express";
import {
  createUpdateFeedbackCollection,
  getFeedbackCollectionMethod,
  deleteFeedbackCollectionMethod,
} from "../controller/feedbackCollectionMethodController.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { verifyToken } from "../helper/tokenVerify.js";

const feedbackCollectionMethod = Router();

/**
 * @swagger
 * tags:
 *   name: Feedback Collection Methods
 *   description: API operations for managing feedback collection methods
 */

/**
 * @swagger
 * /api/feedbackCollection/create-update-feedback-collection:
 *   post:
 *     summary: Create or update a feedback collection method
 *     tags: [Feedback Collection Methods]
 *     security:
 *       - bearerAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: integer
 *                 example: 1
 *               method_name:
 *                 type: string
 *                 example: "Online Survey"
 *               method_type:
 *                 type: string
 *                 example: "Survey"
 *               review_date:
 *                 type: string
 *                 format: date
 *                 example: "2024-07-30"
 *               collection_method_description:
 *                 type: string
 *                 example: "Detailed description of the method."
 *               target_stakeholder:
 *                 type: string
 *                 example: "Employees"
 *               frequency:
 *                 type: string
 *                 example: "Monthly"
 *               responsible_person:
 *                 type: integer
 *                 example: 5
 *               organization:
 *                 type: integer
 *                 example: 1
 *               department:
 *                 type: integer
 *                 example: 2
 *               deleted:
 *                 type: integer
 *                 example: 0
 *               created_at:
 *                 type: string
 *                 format: date-time
 *               created_by:
 *                 type: integer
 *                 example: 1
 *               updated_at:
 *                 type: string
 *                 format: date-time
 *               updated_by:
 *                 type: integer
 *                 example: 1
 *     responses:
 *       200:
 *         description: Feedback collection method created or updated successfully
 *       400:
 *         description: Bad request
 */
feedbackCollectionMethod.post(
  "/create-update-feedback-collection",
  verifyToken,
  controllerWrapper(createUpdateFeedbackCollection)
);

/**
 * @swagger
 * /api/feedbackCollection/get-feedback-collection:
 *   get:
 *     summary: Get all feedback collection methods
 *     tags: [Feedback Collection Methods]
 *     security:
 *       - bearerAuth: []
 *     responses:
 *       200:
 *         description: A list of feedback collection methods
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 type: object
 *                 properties:
 *                   id:
 *                     type: integer
 *                     example: 1
 *                   method_name:
 *                     type: string
 *                     example: "Online Survey"
 *                   method_type:
 *                     type: string
 *                     example: "Survey"
 *                   review_date:
 *                     type: string
 *                     format: date
 *                     example: "2024-07-30"
 *                   collection_method_description:
 *                     type: string
 *                     example: "Detailed description of the method."
 *                   target_stakeholder:
 *                     type: string
 *                     example: "Employees"
 *                   frequency:
 *                     type: string
 *                     example: "Monthly"
 *                   responsible_person:
 *                     type: integer
 *                     example: 5
 *                   organization:
 *                     type: integer
 *                     example: 1
 *                   department:
 *                     type: integer
 *                     example: 2
 *                   deleted:
 *                     type: integer
 *                     example: 0
 *                   created_at:
 *                     type: string
 *                     format: date-time
 *                   created_by:
 *                     type: integer
 *                     example: 1
 *                   updated_at:
 *                     type: string
 *                     format: date-time
 *                   updated_by:
 *                     type: integer
 *                     example: 1
 *       400:
 *         description: Bad request
 */
feedbackCollectionMethod.get(
  "/get-feedback-collection",
  verifyToken,
  controllerWrapper(getFeedbackCollectionMethod)
);

/**
 * @swagger
 * /api/feedbackCollection/get-feedback-collection/{id}:
 *   get:
 *     summary: Get a feedback collection method by ID
 *     tags: [Feedback Collection Methods]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The ID of the feedback collection method
 *     responses:
 *       200:
 *         description: A feedback collection method
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 id:
 *                   type: integer
 *                   example: 1
 *                 method_name:
 *                   type: string
 *                   example: "Online Survey"
 *                 method_type:
 *                   type: string
 *                   example: "Survey"
 *                 review_date:
 *                   type: string
 *                   format: date
 *                   example: "2024-07-30"
 *                 collection_method_description:
 *                   type: string
 *                   example: "Detailed description of the method."
 *                 target_stakeholder:
 *                   type: string
 *                   example: "Employees"
 *                 frequency:
 *                   type: string
 *                   example: "Monthly"
 *                 responsible_person:
 *                   type: integer
 *                   example: 5
 *                 organization:
 *                   type: integer
 *                   example: 1
 *                 department:
 *                   type: integer
 *                   example: 2
 *                 deleted:
 *                   type: integer
 *                   example: 0
 *                 created_at:
 *                   type: string
 *                   format: date-time
 *                 created_by:
 *                   type: integer
 *                   example: 1
 *                 updated_at:
 *                   type: string
 *                   format: date-time
 *                 updated_by:
 *                   type: integer
 *                   example: 1
 *       400:
 *         description: Bad request
 *       404:
 *         description: Feedback collection method not found
 */
feedbackCollectionMethod.get(
  "/get-feedback-collection/:id",
  verifyToken,
  controllerWrapper(getFeedbackCollectionMethod)
);

/**
 * @swagger
 * /api/feedbackCollection/delete-feedback-collection/{id}:
 *   delete:
 *     summary: Delete a feedback collection method by ID
 *     tags: [Feedback Collection Methods]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The ID of the feedback collection method to delete
 *     responses:
 *       200:
 *         description: Feedback collection method deleted successfully
 *       400:
 *         description: Bad request
 *       404:
 *         description: Feedback collection method not found
 */
feedbackCollectionMethod.delete(
  "/delete-feedback-collection/:id",
  verifyToken,
  controllerWrapper(deleteFeedbackCollectionMethod)
);

export default feedbackCollectionMethod;
