// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  createUpdateDocumentType,
  deleteDocumentType,
  getParentDocumentTypeList,
  viewDocumentType,
} from "../controller/documentTypeController.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { validateRequest } from "../helper/general.js";
import { documentTypeCreateUpdateValidationSchema } from "../validationsRecord/validationSchema.js";

/**
 * @swagger
 * tags:
 *   name: Document Type
 *   description: API operations for managing all Document Types
 */
const documentTypeRoutes = Router();

/**
 * @swagger
 * /api/documentType/create-update-document-type:
 *   post:
 *     summary: Create or update a Document Type
 *     tags: [Document Type]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: string
 *               name:
 *                 type: string
 *               description:
 *                 type: string
 *               organization:
 *                 type: array
 *                 items:
 *                   type: string
 *           example: {
 *             id: "id",
 *             name: "name",
 *             description: "description",
 *             organization: []
 *           }
 */
documentTypeRoutes.post(
  "/create-update-document-type",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      146,
      "document_type"
    )(req, res, next);
  },
  validateRequest(documentTypeCreateUpdateValidationSchema),
  controllerWrapper(createUpdateDocumentType)
);

/**
 * @swagger
 * /api/documentType/get-document-type:
 *   get:
 *     summary: Get a list of all Document Types
 *     tags: [Document Type]
 */
documentTypeRoutes.get(
  "/get-document-type",
  verifyToken,
  sideBarPermissionCheck("view", 146, "document_type"),
  controllerWrapper(viewDocumentType)
);

/**
 * @swagger
 * /api/documentType/get-document-type/{id}:
 *   get:
 *     summary: Get a specific record of Document Type
 *     tags: [Document Type]
 *     parameters:
 *       - name: id
 *         in: path
 *         required: true
 *         description: ID of the Document Type to retrieve
 *         schema:
 *           type: string
 */
documentTypeRoutes.get(
  "/get-document-type/:id",
  verifyToken,
  sideBarPermissionCheck("view", 146, "document_type"),
  controllerWrapper(viewDocumentType)
);

/**
 * @swagger
 * /api/documentType/delete-document-type/{id}:
 *   delete:
 *     summary: Delete a specific record of Document Type
 *     tags: [Document Type]
 *     parameters:
 *       - name: id
 *         in: path
 *         required: true
 *         description: ID of the Document Type to delete
 *         schema:
 *           type: string
 */
documentTypeRoutes.delete(
  "/delete-document-type/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 146, "document_type"),
  controllerWrapper(deleteDocumentType)
);

/**
 * @swagger
 * /api/documentType/get-document-type-list:
 *   get:
 *     summary: Get the list for selecting parent category
 *     tags: [Document Type]
 *     parameters:
 *       - in: query
 *         organization : [1,2,34,4]
 *         schema:
 *           type: Array
 *         description: Array of organization to fetch the list
 */

documentTypeRoutes.get(
  "/get-document-type-list",
  verifyToken,
  controllerWrapper(getParentDocumentTypeList)
);


export default documentTypeRoutes;
