// Import necessary modules
import { Router } from "express";
import {
  createUpdateDocument,
  deleteDocumentCreation,
  documentDownloadLog,
  getDocument,
  getDocumentOverviewData,
  getDocumentWithVersions,
  getExpiringDocument,
  shareDocument,
  updateDocumentStatus,
} from "../controller/documentCreationController.js";
import { verifyToken } from "../helper/tokenVerify.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { getUserAndDocumentDropdown } from "../controller/auditLogController.js";

/**
 * @swagger
 * tags:
 *   name: DocumentCreation
 *   description: API operations for managing documents in the system
 */
const DocumentCreationRoutes = Router();
/**
 * @swagger
 * /api/documentCreation/create-update-document:
 *   post:
 *     summary: Create or update a document
 *     tags: [DocumentCreation]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: integer
 *                 description: Unique ID of the document (used for updates)
 *               document_amendment:
 *                 type: string
 *                 description: Details about document amendment
 *               select_revised_document:
 *                 type: string
 *                 description: Reference to the revised document
 *               document_name:
 *                 type: string
 *                 description: Name of the document
 *               unique_id:
 *                 type: string
 *                 description: Unique identifier for the document
 *               document_type:
 *                 type: string
 *                 description: Type/category of the document
 *               description:
 *                 type: string
 *                 description: Detailed description of the document
 *               upload_document:
 *                 type: string
 *                 format: binary
 *                 description: File to be uploaded
 *               version_number:
 *                 type: string
 *                 description: Version number of the document
 *               document_owner:
 *                 type: string
 *                 description: Owner of the document
 *               document_hierarchy:
 *                 type: string
 *                 description: Document hierarchy
 *               document_classification:
 *                 type: string
 *                 description: Classification of the document
 *               document_tags:
 *                 type: string
 *                 description: Tags associated with the document
 *               viewing_rights_employees:
 *                 type: string
 *                 description: Employees with viewing rights
 *               modification_rights_employees:
 *                 type: string
 *                 description: Employees with modification rights
 *               deleting_rights_employees:
 *                 type: string
 *                 description: Employees with deleting rights
 *               effective_date:
 *                 type: string
 *                 format: date-time
 *                 description: Document effective date
 *               expiry_date:
 *                 type: string
 *                 format: date-time
 *                 description: Document expiry date
 *               document_keywords:
 *                 type: string
 *                 description: Keywords for the document
 *               employee_reviewer:
 *                 type: string
 *                 description: Reviewer's employee details
 *               reviewer_employee_name:
 *                 type: integer
 *                 description: Name of the reviewer employee (as ID)
 *               reviewer_role:
 *                 type: string
 *                 description: Role of the reviewer
 *               reviewer_department:
 *                 type: string
 *                 description: Department of the reviewer
 *               approver_employee_name:
 *                 type: integer
 *                 description: Name of the approver employee (as ID)
 *               approver_role:
 *                 type: string
 *                 description: Role of the approver
 *               approver_department:
 *                 type: string
 *                 description: Department of the approver
 *               document_status:
 *                 type: string
 *                 description: Current status of the document
 *               organization:
 *                 type: integer
 *                 description: Organization ID associated with the document
 *               department:
 *                 type: integer
 *                 description: Department ID associated with the document
 *           example:
 *             id: 1
 *             document_amendment: "Revised for version update"
 *             select_revised_document: "Doc_1234"
 *             document_name: "New Policy Document"
 *             unique_id: "DOC-001-2024"
 *             document_type: "Policy"
 *             description: "This document outlines new company policies"
 *             upload_document: "binary file data"
 *             version_number: "v2.0"
 *             document_owner: "Admin"
 *             document_hierarchy: "Level 1 > Level 2"
 *             document_classification: "Confidential"
 *             document_tags: "Policy, HR"
 *             viewing_rights_employees: "Employee IDs"
 *             modification_rights_employees: "Admin IDs"
 *             deleting_rights_employees: "Super Admin"
 *             effective_date: "2024-01-01T00:00:00Z"
 *             expiry_date: "2025-01-01T00:00:00Z"
 *             document_keywords: "Policy, Guidelines"
 *             employee_reviewer: "Employee 1"
 *             reviewer_employee_name: 101
 *             reviewer_role: "Reviewer"
 *             reviewer_department: "HR"
 *             approver_employee_name: 201
 *             approver_role: "Approver"
 *             approver_department: "Admin"
 *             document_status: "Active"
 *             organization: 14
 *             department: 5
 *     responses:
 *       200:
 *         description: Document created or updated successfully
 *       400:
 *         description: Bad request. Invalid input parameters
 *       500:
 *         description: Internal server error
 */

DocumentCreationRoutes.post("/create-update-document", verifyToken, controllerWrapper(createUpdateDocument));

/**
 * @swagger
 * /api/documentCreation/get-document:
 *   get:
 *     summary: Get a list of all documents
 *     tags: [DocumentCreation]
 *     responses:
 *       200:
 *         description: List of documents retrieved successfully
 *       500:
 *         description: Internal server error
 */
DocumentCreationRoutes.get("/get-document", verifyToken, controllerWrapper(getDocument));

/**
 * @swagger
 * /api/documentCreation/get-document/{id}:
 *   get:
 *     summary: Get a specific document by ID
 *     tags: [DocumentCreation]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: string
 *         required: true
 *         description: The ID of the document to retrieve
 *     responses:
 *       200:
 *         description: Document data retrieved successfully
 *       404:
 *         description: Document not found
 *       500:
 *         description: Internal server error
 */
DocumentCreationRoutes.get("/get-document/:id", verifyToken, controllerWrapper(getDocument));

/**
 * @swagger
 * /api/documentCreation/get-document-history/{id}:
 *   get:
 *     summary: Get a specific document by ID
 *     tags: [DocumentCreation]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: string
 *         required: true
 *         description: The ID of the document to retrieve
 *     responses:
 *       200:
 *         description: Document data retrieved successfully
 *       404:
 *         description: Document not found
 *       500:
 *         description: Internal server error
 */
DocumentCreationRoutes.get("/get-document-history/:id", verifyToken, controllerWrapper(getDocumentWithVersions));

/**
 * @swagger
 * /api/documentCreation/delete-document/{id}:
 *   delete:
 *     summary: Delete a specific document by ID
 *     tags: [DocumentCreation]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: string
 *         required: true
 *         description: The ID of the document to delete
 *     responses:
 *       200:
 *         description: Document successfully deleted
 *       404:
 *         description: Document not found
 *       500:
 *         description: Internal server error
 */
DocumentCreationRoutes.delete("/delete-document/:id", verifyToken, controllerWrapper(deleteDocumentCreation));

DocumentCreationRoutes.get("/document-overview-data", verifyToken, controllerWrapper(getDocumentOverviewData));

/**
 * @swagger
 * /api/documentCreation/share-document:
 *   post:
 *     summary: Get a list of all documents
 *     tags: [DocumentCreation]
 *     responses:
 *       200:
 *         description: List of documents retrieved successfully
 *       500:
 *         description: Internal server error
 */
DocumentCreationRoutes.post("/share-document", verifyToken, controllerWrapper(shareDocument));

/**
 * @swagger
 * /api/documentCreation/download-document:
 *   post:
 *     summary: Download Document
 *     tags: [DocumentCreation]
 *     responses:
 *       200:
 *         description: Download document and add log in audit log
 *       500:
 *         description: Internal server error
 */
DocumentCreationRoutes.post("/download-document", verifyToken, controllerWrapper(documentDownloadLog));

/**
 * @swagger
 * /api/documentCreation/expiring-document:
 *   get:
 *     summary: Get a list of all documents
 *     tags: [DocumentCreation]
 *     responses:
 *       200:
 *         description: List of documents retrieved successfully
 *       500:
 *         description: Internal server error
 */
DocumentCreationRoutes.get("/expiring-document", verifyToken, controllerWrapper(getExpiringDocument));
/**
 * @swagger
 * /api/documentCreation/users-document-dropdown:
 *   get:
 *     summary: Get a list of all documents and users
 *     tags: [DocumentCreation]
 *     responses:
 *       200:
 *         description: List of documents retrieved successfully
 *       500:
 *         description: Internal server error
 */
DocumentCreationRoutes.get("/users-document-dropdown", verifyToken, controllerWrapper(getUserAndDocumentDropdown));

/**
 * @swagger
 * /api/documentCreation/update-status:
 *   post:
 *     summary: Update document status
 *     tags: [DocumentCreation]
 *     responses:
 *       200:
 *         description: Document status updated
 *       500:
 *         description: Internal server error
 */
DocumentCreationRoutes.post("/update-status", verifyToken, controllerWrapper(updateDocumentStatus));

export default DocumentCreationRoutes;
