// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import {
  createUpdateDisability,
  deleteDisability,
  getParentDisabilityList,
  viewDisability,
  viewDisabilityDropDown,
} from "../controller/disabilityController.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import { controllerWrapper } from "../helper/wrapper.js";

/**
 * @swagger
 * tags:
 *   name: Disability
 *   description: API operations for managing all Disability
 */
const disabilityRoutes = Router();

/**
 * @swagger
 * /api/disability/create-update-disability:
 *   post:
 *     summary: Create or update an Disability
 *     tags: [Disability]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: string
 *               name:
 *                 type: string
 *               description:
 *                 type: string
 *               organization:
 *                 type: array
 *                 items:
 *                   type: string
 *           example:
 *             id: "id"
 *             name: "Policy Impact Name"
 *             description: "Description of the disability"
 *             organization: ["Organization1", "Organization2"]
 *     responses:
 *       200:
 *         description: Disability created or updated successfully
 *       400:
 *         description: Bad request. Invalid input parameters
 *       500:
 *         description: Internal server error
 */
disabilityRoutes.post(
  "/create-update-disability",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(req.body.id ? "edit" : "create", 341, "disability")(req, res, next);
  },
  controllerWrapper(createUpdateDisability)
);

/**
 * @swagger
 * /api/disability/get-disability:
 *   get:
 *     summary: Get list of all Disability
 *     tags: [Disability]
 *     responses:
 *       200:
 *         description: List of disability
 *       500:
 *         description: Internal server error
 */
disabilityRoutes.get(
  "/get-disability",
  verifyToken,
  sideBarPermissionCheck("view", 341, "disability"),
  controllerWrapper(viewDisability)
);

/**
 * @swagger
 * /api/disability/get-disability/{id}:
 *   get:
 *     summary: Get a specific record of Disability
 *     tags: [Disability]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: string
 *         required: true
 *         description: The ID of the disability
 *     responses:
 *       200:
 *         description: Disability data
 *       404:
 *         description: Disability not found
 *       500:
 *         description: Internal server error
 */
disabilityRoutes.get(
  "/get-disability/:id",
  verifyToken,
  // sideBarPermissionCheck("view", 341, "disability"),
  controllerWrapper(viewDisability)
);
/**
 * @swagger
 * /api/disability/get-disability-dropdown:
 *   get:
 *     summary: Get the list all Disability
 *     tags:
 *       - Disability
 *     description: The disability
 *     responses:
 *       200:
 *         description: Disability data
 *       404:
 *         description: Disability not found
 *       500:
 *         description: Internal server error
 */
disabilityRoutes.get(
  "/get-disability-dropdown",
  verifyToken,
  // sideBarPermissionCheck("view", 341, "disability"),
  controllerWrapper(viewDisabilityDropDown)
);

/**
 * @swagger
 * /api/incidentPpe/get-disability-list:
 *   get:
 *     summary: Get the list for selecting parent category
 *     tags: [Disability]
 *     parameters:
 *       - in: query
 *         organization : [1,2,34,4]
 *         schema:
 *           type: Array
 *         description: Array of organization to fetch the list
 */

disabilityRoutes.get("/get-disability-list", verifyToken, controllerWrapper(getParentDisabilityList));

/**
 * @swagger
 * /api/disability/delete-disability/{id}:
 *   delete:
 *     summary: Delete a specific record of Disability
 *     tags: [Disability]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: string
 *         required: true
 *         description: The ID of the disability to delete
 *     responses:
 *       200:
 *         description: Disability successfully deleted
 *       404:
 *         description: Disability not found
 *       500:
 *         description: Internal server error
 */
disabilityRoutes.delete(
  "/delete-disability/:id",
  verifyToken,
  // sideBarPermissionCheck("delete", 341, "disability"),
  controllerWrapper(deleteDisability)
);

export default disabilityRoutes;
