import { Router } from "express";
import {
  createUpdateContractorRegistration,
  getContractorRegistration,
  deleteContractorRegistration,
} from "../controller/contractorRegistrationController.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";

const contractorRegistrationRoutes = Router();

/**
 * @swagger
 * tags:
 *   name: Contractor Registration
 *   description: API operations for managing contractor registrations
 */
/**
 * @swagger
 * /api/contractorRegistration/create-update-contractor-registration:
 *   post:
 *     summary: Create or update a contractor registration
 *     tags: [Contractor Registration]
 *     security:
 *       - bearerAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               contractor_name:
 *                 type: string
 *                 example: "ABC Contractors Ltd"
 *               phone_number:
 *                 type: string
 *                 example: "123-456-7890"
 *               email:
 *                 type: string
 *                 example: "contractor@mailues.com"
 *               physical_address:
 *                 type: string
 *                 example: "House No. 420 white house"
 *               cipc_registration_number:
 *                 type: string
 *                 example: "2024/123456/07"
 *               tax_number:
 *                 type: string
 *                 example: "1234567890"
 *               bbbee_status:
 *                 type: string
 *                 example: "Level 1"
 *               vat_number:
 *                 type: string
 *                 example: "123456789"
 *               bank_name:
 *                 type: string
 *                 example: "XYZ Bank"
 *               account_number:
 *                 type: string
 *                 example: "123456789"
 *               bank_code:
 *                 type: string
 *                 example: "12345"
 *               account_type:
 *                 type: string
 *                 example: "Savings"
 *               payment_terms:
 *                 type: string
 *                 example: "30 days"
 *               type_of_work:
 *                 type: string
 *                 example: "Construction"
 *               start_date:
 *                 type: string
 *                 format: date
 *                 example: "2024-08-01"
 *               end_date:
 *                 type: string
 *                 format: date
 *                 example: "2025-07-31"
 *               contract_value:
 *                 type: string
 *                 example: "1000000"
 *               currency:
 *                 type: string
 *                 example: "USD"
 *               allocation_human_resource:
 *                 type: string
 *                 example: "Details about human resource allocation."
 *               classification_value:
 *                 type: string
 *                 example: "High"
 *               classification_criticality:
 *                 type: string
 *                 example: "Critical"
 *               classification_impact:
 *                 type: string
 *                 example: "High impact"
 *               classification_category:
 *                 type: string
 *                 example: "Category A"
 *               employee_name:
 *                 type: string
 *                 example: "Jane"
 *               employee_surname:
 *                 type: string
 *                 example: "Doe"
 *               employee_id_passport:
 *                 type: string
 *                 example: "AB123456"
 *               employee_designation:
 *                 type: string
 *                 example: "Site Manager"
 *               organization:
 *                 type: integer
 *                 example: 2
 *               department:
 *                 type: integer
 *                 example: 3
 *               deleted:
 *                 type: integer
 *                 example: 0
 *               created_by:
 *                 type: integer
 *                 example: 1
 *               created_at:
 *                 type: string
 *                 format: date-time
 *                 example: "2024-08-01T00:00:00.000Z"
 *               updated_by:
 *                 type: integer
 *                 example: 2
 *               updated_at:
 *                 type: string
 *                 format: date-time
 *                 example: "2024-09-01T00:00:00.000Z"
 *     responses:
 *       200:
 *         description: Contractor registration created or updated successfully
 *       400:
 *         description: Bad request
 */

contractorRegistrationRoutes.post(
  "/create-update-contractor-registration",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      254,
      "contractor_registration"
    )(req, res, next);
  },
  controllerWrapper(createUpdateContractorRegistration)
);

/**
 * @swagger
 * /api/contractorRegistration/get-contractor-registration:
 *   get:
 *     summary: Get all contractor registrations
 *     tags: [Contractor Registration]
 *     security:
 *       - bearerAuth: []
 *     responses:
 *       200:
 *         description: A list of contractor registrations
 *       400:
 *         description: Bad request
 */
contractorRegistrationRoutes.get(
  "/get-contractor-registration",
  verifyToken,
  sideBarPermissionCheck("view", 254, "contractor_registration"),
  controllerWrapper(getContractorRegistration)
);

/**
 * @swagger
 * /api/contractorRegistration/get-contractor-registration/{id}:
 *   get:
 *     summary: Get a contractor registration by ID
 *     tags: [Contractor Registration]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The ID of the contractor registration
 *     responses:
 *       200:
 *         description: A contractor registration
 *       400:
 *         description: Bad request
 */
contractorRegistrationRoutes.get(
  "/get-contractor-registration/:id",
  verifyToken,
  sideBarPermissionCheck("view", 254, "contractor_registration"),
  controllerWrapper(getContractorRegistration)
);

/**
 * @swagger
 * /api/contractorRegistration/delete-contractor-registration/{id}:
 *   delete:
 *     summary: Delete a contractor registration by ID
 *     tags: [Contractor Registration]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The ID of the contractor registration
 *     responses:
 *       200:
 *         description: Contractor registration deleted successfully
 *       400:
 *         description: Bad request
 */
contractorRegistrationRoutes.delete(
  "/delete-contractor-registration/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 254, "contractor_registration"),
  controllerWrapper(deleteContractorRegistration)
);

export default contractorRegistrationRoutes;
