import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  createUpdateContractorQuestionnaire,
  getContractorQuestionnaire,
  deleteContractorQuestionnaire,
  sendQuestionnaire,
  fillForm,
} from "../controller/contractorQuestionnaireController.js";
import { controllerWrapper } from "../helper/wrapper.js";

/**
 * @swagger
 * tags:
 *   name: Contractor Questionnaire
 *   description: API operations for managing contractor questionnaires
 */

/**
 * @swagger
 * components:
 *   schemas:
 *     ContractorQuestionnaire:
 *       type: object
 *       properties:
 *         id:
 *           type: integer
 *           description: The auto-generated ID of the contractor questionnaire
 *         contractor:
 *           type: integer
 *           description: ID of the contractor
 *         status:
 *           type: string
 *           description: The current status of the questionnaire
 *         form:
 *           type: string
 *           description: The form data in long text format
 *         public_link:
 *           type: string
 *           description: Public link for the questionnaire
 *         images:
 *           type: string
 *           description: Associated images in long text format
 *         organization:
 *           type: integer
 *           description: ID of the associated organization
 *         department:
 *           type: integer
 *           description: ID of the associated department
 *         deleted:
 *           type: integer
 *           description: Whether the questionnaire is deleted (0 or 1)
 *         created_by:
 *           type: integer
 *           description: ID of the user who created the questionnaire
 *         created_at:
 *           type: string
 *           format: date-time
 *           description: The date and time when the questionnaire was created
 *         updated_by:
 *           type: integer
 *           description: ID of the user who last updated the questionnaire
 *         updated_at:
 *           type: string
 *           format: date-time
 *           description: The date and time when the questionnaire was last updated
 */

const contractorQuestionnaireRoutes = Router();

/**
 * @swagger
 * /api/contractorQuestionnaire/create-update-contractor-questionnaire:
 *   post:
 *     summary: Create or update a contractor questionnaire
 *     tags: [Contractor Questionnaire]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/ContractorQuestionnaire'
 *     responses:
 *       200:
 *         description: The contractor questionnaire was successfully created/updated
 */

contractorQuestionnaireRoutes.post(
  "/create-update-contractor-questionnaire",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      257,
      "contractor_questionnaire"
    )(req, res, next);
  },
  controllerWrapper(createUpdateContractorQuestionnaire)
);

/**
 * @swagger
 * /api/contractorQuestionnaire/get-contractor-questionnaire:
 *   get:
 *     summary: Get all contractor questionnaires
 *     tags: [Contractor Questionnaire]
 *     responses:
 *       200:
 *         description: List of all contractor questionnaires
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/ContractorQuestionnaire'
 */

contractorQuestionnaireRoutes.get(
  "/get-contractor-questionnaire",
  verifyToken,
  sideBarPermissionCheck("view", 257, "contractor_questionnaire"),
  controllerWrapper(getContractorQuestionnaire)
);

/**
 * @swagger
 * /api/contractorQuestionnaire/get-contractor-questionnaire/{id}:
 *   get:
 *     summary: Get a contractor questionnaire by ID
 *     tags: [Contractor Questionnaire]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The ID of the contractor questionnaire
 *     responses:
 *       200:
 *         description: Contractor questionnaire data
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ContractorQuestionnaire'
 */

contractorQuestionnaireRoutes.get(
  "/get-contractor-questionnaire/:id",
  verifyToken,
  sideBarPermissionCheck("view", 257, "contractor_questionnaire"),
  controllerWrapper(getContractorQuestionnaire)
);

/**
 * @swagger
 * /api/contractorQuestionnaire/delete-contractor-questionnaire/{id}:
 *   delete:
 *     summary: Delete a contractor questionnaire by ID
 *     tags: [Contractor Questionnaire]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The ID of the contractor questionnaire to delete
 *     responses:
 *       200:
 *         description: Contractor questionnaire was deleted
 */

contractorQuestionnaireRoutes.delete(
  "/delete-contractor-questionnaire/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 257, "contractor_questionnaire"),
  controllerWrapper(deleteContractorQuestionnaire)
);

/**
 * @swagger
 * /api/contractorQuestionnaire/send-questionnaire:
 *   post: 
 *     summary: Send a contractor questionnaire
 *     tags: [Contractor Questionnaire]
 *     responses:
 *       200:
 *         description: The questionnaire was successfully sent
 */

contractorQuestionnaireRoutes.post(
  "/send-questionnaire",
  verifyToken,
  sideBarPermissionCheck("view", 257, "contractor_questionnaire"),
  controllerWrapper(sendQuestionnaire)
);

/**
 * @swagger
 * /api/contractorQuestionnaire/fill-questionnaire:
 *   post:
 *     summary: Fill out a contractor questionnaire form
 *     tags: [Contractor Questionnaire]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/ContractorQuestionnaire'
 *     responses:
 *       200:
 *         description: The form was successfully submitted
 */

contractorQuestionnaireRoutes.post(
  "/fill-questionnaire",
  controllerWrapper(fillForm)
);

export default contractorQuestionnaireRoutes;
