// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  createUpdatedContractorPermitLicense,
  deleteContractorPermitLicense,
  viewAllContractorPermitLicense,
} from "../controller/contractorPermitLicenseController.js";
import { controllerWrapper } from "../helper/wrapper.js";
/**
 * @swagger
 * tags:
 *   name: Contractor Permit License
 *   description: API operations for managing all Contractor Permit Licenses
 */

const contractorPermitLicenseRoutes = Router();

/**
 * @swagger
 * /api/contractorPermitLicense/create-update-contractor-permit-license:
 *   post:
 *     summary: Create or update a Contractor Permit License
 *     tags: [Contractor Permit License]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *           example:
 *             id: 1234
 *             title: 5678
 *             description: "Permit for operating a waste management facility"
 *             permit_holder: 7890
 *             permit_license_number: "PERMIT-2022-123"
 *             permit_type: "Waste Management"
 *             issuing_authority: 13579
 *             permit_license_id: "ABC12345XYZ"
 *             applicable_section_of_legislation: "Environmental Protection Act"
 *             issue_date: "2022-01-15"
 *             validity_from: "2022-02-01"
 *             validity_until: "2023-01-31"
 *             notification_alert: "30 days before expiry"
 *             department: "Environmental Department"
 *             organization: 2468
 *             created_by: 1
 *             updated_by: 2
 *     responses:
 *       200:
 *         description: Contractor Permit License created or updated successfully
 *       400:
 *         description: Bad request
 */

contractorPermitLicenseRoutes.post(
  "/create-update-contractor-permit-license",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      267,
      "contractor_permit_license"
    )(req, res, next);
  },
  controllerWrapper(createUpdatedContractorPermitLicense)
);

/**
 * @swagger
 * /api/contractorPermitLicense/get-contractor-permit-license:
 *   get:
 *     summary: Get all Contractor Permit Licenses
 *     tags: [Contractor Permit License]
 *     responses:
 *       200:
 *         description: A list of Contractor Permit Licenses
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 type: object
 *                 properties:
 *                   id:
 *                     type: integer
 *                   title:
 *                     type: integer
 *                   description:
 *                     type: string
 *                   permit_holder:
 *                     type: integer
 *                   permit_license_number:
 *                     type: string
 *                   permit_type:
 *                     type: string
 *                   issuing_authority:
 *                     type: integer
 *                   permit_license_id:
 *                     type: string
 *                   applicable_section_of_legislation:
 *                     type: string
 *                   issue_date:
 *                     type: string
 *                     format: date
 *                   validity_from:
 *                     type: string
 *                     format: date
 *                   validity_until:
 *                     type: string
 *                     format: date
 *                   notification_alert:
 *                     type: string
 *                   department:
 *                     type: string
 *                   organization:
 *                     type: integer
 *                   created_by:
 *                     type: integer
 *                   updated_by:
 *                     type: integer
 *                   created_at:
 *                     type: string
 *                     format: date-time
 *                   updated_at:
 *                     type: string
 *                     format: date-time
 *       400:
 *         description: Bad request
 */

contractorPermitLicenseRoutes.get(
  "/get-contractor-permit-license",
  verifyToken,
  sideBarPermissionCheck("view", 267, "contractor_permit_license"),
  controllerWrapper(viewAllContractorPermitLicense)
);

/**
 * @swagger
 * /api/contractorPermitLicense/get-contractor-permit-license/{id}:
 *   get:
 *     summary: Get a specific Contractor Permit License by ID
 *     tags: [Contractor Permit License]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The ID of the Contractor Permit License
 *     responses:
 *       200:
 *         description: A Contractor Permit License object
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 id:
 *                   type: integer
 *                 title:
 *                   type: integer
 *                 description:
 *                   type: string
 *                 permit_holder:
 *                   type: integer
 *                 permit_license_number:
 *                   type: string
 *                 permit_type:
 *                   type: string
 *                 issuing_authority:
 *                   type: integer
 *                 permit_license_id:
 *                   type: string
 *                 applicable_section_of_legislation:
 *                   type: string
 *                 issue_date:
 *                   type: string
 *                   format: date
 *                 validity_from:
 *                   type: string
 *                   format: date
 *                 validity_until:
 *                   type: string
 *                   format: date
 *                 notification_alert:
 *                   type: string
 *                 department:
 *                   type: string
 *                 organization:
 *                   type: integer
 *                 created_by:
 *                   type: integer
 *                 updated_by:
 *                   type: integer
 *                 created_at:
 *                   type: string
 *                   format: date-time
 *                 updated_at:
 *                   type: string
 *                   format: date-time
 *       400:
 *         description: Bad request
 */

contractorPermitLicenseRoutes.get(
  "/get-contractor-permit-license/:id",
  verifyToken,
  sideBarPermissionCheck("view", 267, "contractor_permit_license"),
  controllerWrapper(viewAllContractorPermitLicense)
);

/**
 * @swagger
 * /api/contractorPermitLicense/delete-contractor-permit-license/{id}:
 *   delete:
 *     summary: Delete a specific Contractor Permit License by ID
 *     tags: [Contractor Permit License]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The ID of the Contractor Permit License
 *     responses:
 *       200:
 *         description: Contractor Permit License deleted successfully
 *       400:
 *         description: Bad request
 */

contractorPermitLicenseRoutes.delete(
  "/delete-contractor-permit-license/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 267, "contractor_permit_license"),
  controllerWrapper(deleteContractorPermitLicense)
);

export default contractorPermitLicenseRoutes;
