// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  createUpdateContractorInvestigation,
  getContractorInvestigation,
  deleteContractorInvestigation,
} from "../controller/contractorInvestigationController.js";
import { controllerWrapper } from "../helper/wrapper.js";

/**
 * @swagger
 * tags:
 *   name: Incident Investigation
 *   description: API Operations For Managing All Investigations
 */
const ContractorInvestigationRoutes = Router();

/**
 * @swagger
 * /api/investigation/create-update-investigation:
 *   post:
 *     summary: Create or Update an Investigation
 *     tags: [Incident Investigation]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example:
 *             id: 1
 *             incident_title: "Incident Title"
 *             recording_location: "Location"
 *             recording_shift: "Day Shift"
 *             recording_day: "2023-01-01"
 *             recording_incident_date: "2023-01-01"
 *             recording_incident_time: "12:00:00"
 *             lead_type: 0
 *             lead_investigator: "Investigator Name"
 *             investigation_team_members: "Member1, Member2"
 *             recording_investigation_start_date: "2023-01-02"
 *             recording_investigation_end_date: "2023-01-05"
 *             recording_description: "Detailed description"
 *             recording_incident_category: "Category"
 *             recording_initial_incident_severity: "High"
 *             recording_type_of_incident: 1
 *             involved_type: 0
 *             involved_person: 1
 *             involved_external_company_name: "External Company"
 *             involved_external_address: "123 Main St"
 *             involved_external_telephone_number: "555-1234"
 *             witnesses: "Witness details"
 *             supervisor_type: 0
 *             supervisor_person: 1
 *             environment_consequence_type: "Pollution"
 *             environment_pollution: "Details of pollution"
 *             health_consequence_type: "Injury"
 *             health_exposure: "Exposure details"
 *             safety_consequence_type: "Accident"
 *             safety_exposure: "Exposure details"
 *             safety_injuries: "Injury details"
 *             safety_fatality: "No fatalities"
 *             legal_consequence_type: "Legal action"
 *             assets_damage_assets_property_type: "Property type"
 *             security_consequence_type: "Breach"
 *             finance_consequence_type: "Financial impact"
 *             operations_consequence_type: "Operational delay"
 *             investigation_immediate_causes: "Causes"
 *             investigation_root_cause: "Root cause"
 *             action_plans: "Action plans"
 *             investigation_lead: 1
 *             investigation_contact_no: "555-5678"
 *             investigation_date: "2023-01-01"
 *             she_representative_lead: 1
 *             she_representative_contact_no: "555-8765"
 *             she_representative_date: "2023-01-01"
 *             line_manager_type: 0
 *             line_manager_action_plans: "Action plans"
 *             any_other_comments: "Additional comments"
 *             reportableData: "Reportable data"
 *             department: 1
 *             organization: 1
 *             created_by: 1
 *             created_at: "2023-01-01T12:00:00Z"
 *             updated_at: "2023-01-01T12:00:00Z"
 *             deleted: 0
 *     responses:
 *       200:
 *         description: Investigation created or updated successfully
 *       400:
 *         description: Bad request
 *       500:
 *         description: Internal server error
 */
ContractorInvestigationRoutes.post(
  "/create-update-contractor-investigation",
  verifyToken,
  // (req, res, next) => {
  //   sideBarPermissionCheck(
  //     req.body.id ? "edit" : "create",
  //     176,
  //     "investigation"
  //   )(req, res, next);
  // },
  controllerWrapper(createUpdateContractorInvestigation)
);

/**
 * @swagger
 * /api/investigation/get-investigation:
 *   get:
 *     summary: Get All Investigation List
 *     tags: [Incident Investigation]
 *     responses:
 *       200:
 *         description: A list of investigations
 *       401:
 *         description: Unauthorized
 *       500:
 *         description: Internal server error
 */
ContractorInvestigationRoutes.get(
  "/get-contractor-investigation",
  verifyToken,
  // sideBarPermissionCheck("view", 176, "contractor_investigation"),
  controllerWrapper(getContractorInvestigation)
);

/**
 * @swagger
 * /api/investigation/get-investigation/{id}:
 *   get:
 *     summary: Get a Specific Investigation Record
 *     tags: [Incident Investigation]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: Investigation ID
 *     responses:
 *       200:
 *         description: A specific investigation record
 *       401:
 *         description: Unauthorized
 *       404:
 *         description: Not found
 *       500:
 *         description: Internal server error
 */
ContractorInvestigationRoutes.get(
  "/get-contractor-investigation/:id",
  verifyToken,
  // sideBarPermissionCheck("view", 176, "contractor_investigation"),
  controllerWrapper(getContractorInvestigation)
);

/**
 * @swagger
 * /api/investigation/delete-investigation/{id}:
 *   delete:
 *     summary: Delete a Specific Investigation Record
 *     tags: [Incident Investigation]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: Investigation ID
 *     responses:
 *       200:
 *         description: Investigation deleted successfully
 *       401:
 *         description: Unauthorized
 *       404:
 *         description: Not found
 *       500:
 *         description: Internal server error
 */
ContractorInvestigationRoutes.delete(
  "/delete-contractor-investigation/:id",
  verifyToken,
  // sideBarPermissionCheck("delete", 176, "contractor_investigation"),
  controllerWrapper(deleteContractorInvestigation)
);

export default ContractorInvestigationRoutes;
