// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  createUpdatedContractorDocumentRepository,
  deleteContractorDocumentRepository,
  viewAllContractorDocumentRepository,
} from "../controller/contractorDocumentRepositoryController.js";
import { validateRequest } from "../helper/general.js";
import { controllerWrapper } from "../helper/wrapper.js";

/**
 * @swagger
 * tags:
 *   name: Contractor Document Repository
 *   description: API operations for managing all Contractor Document Repositories
 */
const contractorDocumentRepositoryRoutes = Router();

/**
 * @swagger
 * /api/contractorDocumentRepository/create-update-contractor-document-repository:
 *   post:
 *     summary: Create or update a Contractor Document Repository
 *     tags: [Contractor Document Repository]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: string
 *               document_type:
 *                 type: string
 *                 enum: [External, Internal]
 *               associated_audit:
 *                 type: string
 *               document_name:
 *                 type: string
 *               document_description:
 *                 type: string
 *               allowed_access_to:
 *                 type: string
 *               audit_execution_id:
 *                 type: string
 *               organization:
 *                 type: array
 *                 items:
 *                   type: string
 *               department:
 *                 type: string
 *           example: {
 *             id: "id",
 *             document_type: "External",
 *             associated_audit: "scheduled_audit_id",
 *             document_name: "document_name",
 *             document_description: "document_description",
 *             allowed_access_to: "allowed_access_to",
 *             audit_execution_id: "audit_execution_id",
 *             organization: [],
 *             department: "department"
 *           }
 */
contractorDocumentRepositoryRoutes.post(
  "/create-update-contractor-document-repository",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      280,
      "contractor_document_repository"
    )(req, res, next);
  },
  // validateRequest(contractorDocumentRepositoryCreateUpdateValidationSchema),
  controllerWrapper(createUpdatedContractorDocumentRepository)
);

/**
 * @swagger
 * /api/contractorDocumentRepository/get-contractor-document-repository:
 *   get:
 *     summary: Get a list of all Contractor Document Repositories
 *     tags: [Contractor Document Repository]
 */
contractorDocumentRepositoryRoutes.get(
  "/get-contractor-document-repository",
  verifyToken,
  sideBarPermissionCheck("view", 280, "contractor_document_repository"),
  controllerWrapper(viewAllContractorDocumentRepository)
);

/**
 * @swagger
 * /api/contractorDocumentRepository/get-contractor-document-repository/{id}:
 *   get:
 *     summary: Get a specific record of Contractor Document Repository
 *     tags: [Contractor Document Repository]
 *     parameters:
 *       - name: id
 *         in: path
 *         required: true
 *         description: ID of the Contractor Document Repository to retrieve
 *         schema:
 *           type: string
 */
contractorDocumentRepositoryRoutes.get(
  "/get-contractor-document-repository/:id",
  verifyToken,
  sideBarPermissionCheck("view", 280, "contractor_document_repository"),
  controllerWrapper(viewAllContractorDocumentRepository)
);

/**
 * @swagger
 * /api/contractorDocumentRepository/delete-contractor-document-repository/{id}:
 *   delete:
 *     summary: Delete a specific record of Contractor Document Repository
 *     tags: [Contractor Document Repository]
 *     parameters:
 *       - name: id
 *         in: path
 *         required: true
 *         description: ID of the Contractor Document Repository to delete
 *         schema:
 *           type: string
 */
contractorDocumentRepositoryRoutes.delete(
  "/delete-contractor-document-repository/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 280, "contractor_document_repository"),
  controllerWrapper(deleteContractorDocumentRepository)
);

export default contractorDocumentRepositoryRoutes;
