import { Router } from "express";
import {
  createUpdateContractorAppointment,
  getContractorAppointment,
  deleteContractorAppointment,
  uploadSignature,
} from "../controller/contractorAppointmentController.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";

const contractorAppointmentRoutes = Router();

/**
 * @swagger
 * tags:
 *   name: Contractor Appointment
 *   description: API operations for managing contractor appointments
 */

/**
 * @swagger
 * /api/contractorAppointment/create-update-contractor-appointment:
 *   post:
 *     summary: Create or update a contractor appointment
 *     tags: [Contractor Appointment]
 *     security:
 *       - bearerAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               who_appoints_name:
 *                 type: string
 *                 example: "John Doe"
 *               legal_appointment_role:
 *                 type: integer
 *                 example: 1
 *               who_is_appointed_name:
 *                 type: string
 *                 example: "Jane Doe"
 *               name_of_appointment:
 *                 type: string
 *                 example: "Site Manager"
 *               effective_date:
 *                 type: string
 *                 format: date
 *                 example: "2024-08-01"
 *               end_date:
 *                 type: string
 *                 format: date
 *                 example: "2025-07-31"
 *               organization:
 *                 type: integer
 *                 example: 2
 *               department:
 *                 type: integer
 *                 example: 3
 *               deleted:
 *                 type: integer
 *                 example: 0
 *               created_by:
 *                 type: integer
 *                 example: 1
 *               created_at:
 *                 type: string
 *                 format: date-time
 *                 example: "2024-08-01T00:00:00.000Z"
 *               updated_by:
 *                 type: integer
 *                 example: 2
 *               updated_at:
 *                 type: string
 *                 format: date-time
 *                 example: "2024-09-01T00:00:00.000Z"
 *     responses:
 *       200:
 *         description: Contractor appointment created or updated successfully
 *       400:
 *         description: Bad request
 */
contractorAppointmentRoutes.post(
  "/create-update-contractor-appointment",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(req.body.id ? "edit" : "create", 256, "contractor_appointments")(req, res, next);
  },
  controllerWrapper(createUpdateContractorAppointment)
);

/**
 * @swagger
 * /api/contractorAppointment/get-contractor-appointment:
 *   get:
 *     summary: Get all contractor appointments
 *     tags: [Contractor Appointment]
 *     security:
 *       - bearerAuth: []
 *     responses:
 *       200:
 *         description: A list of contractor appointments
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 type: object
 *                 properties:
 *                   id:
 *                     type: integer
 *                     example: 1
 *                   who_appoints_name:
 *                     type: string
 *                     example: "John Doe"
 *                   legal_appointment_role:
 *                     type: integer
 *                     example: 1
 *                   who_is_appointed_name:
 *                     type: string
 *                     example: "Jane Doe"
 *                   name_of_appointment:
 *                     type: string
 *                     example: "Site Manager"
 *                   effective_date:
 *                     type: string
 *                     format: date
 *                     example: "2024-08-01"
 *                   end_date:
 *                     type: string
 *                     format: date
 *                     example: "2025-07-31"
 *                   organization:
 *                     type: integer
 *                     example: 2
 *                   department:
 *                     type: integer
 *                     example: 3
 *                   deleted:
 *                     type: integer
 *                     example: 0
 *                   created_by:
 *                     type: integer
 *                     example: 1
 *                   created_at:
 *                     type: string
 *                     format: date-time
 *                     example: "2024-08-01T00:00:00.000Z"
 *                   updated_by:
 *                     type: integer
 *                     example: 2
 *                   updated_at:
 *                     type: string
 *                     format: date-time
 *                     example: "2024-09-01T00:00:00.000Z"
 *       400:
 *         description: Bad request
 */
contractorAppointmentRoutes.get(
  "/get-contractor-appointment",
  verifyToken,
  sideBarPermissionCheck("view", 256, "contractor_appointments"),
  controllerWrapper(getContractorAppointment)
);

/**
 * @swagger
 * /api/contractorAppointment/get-contractor-appointment/{id}:
 *   get:
 *     summary: Get a contractor appointment by ID
 *     tags: [Contractor Appointment]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The ID of the contractor appointment
 *     responses:
 *       200:
 *         description: A contractor appointment
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 id:
 *                   type: integer
 *                   example: 1
 *                 who_appoints_name:
 *                   type: string
 *                   example: "John Doe"
 *                 legal_appointment_role:
 *                   type: integer
 *                   example: 1
 *                 who_is_appointed_name:
 *                   type: string
 *                   example: "Jane Doe"
 *                 name_of_appointment:
 *                   type: string
 *                   example: "Site Manager"
 *                 effective_date:
 *                   type: string
 *                   format: date
 *                   example: "2024-08-01"
 *                 end_date:
 *                   type: string
 *                   format: date
 *                   example: "2025-07-31"
 *                 organization:
 *                   type: integer
 *                   example: 2
 *                 department:
 *                   type: integer
 *                   example: 3
 *                 deleted:
 *                   type: integer
 *                   example: 0
 *                 created_by:
 *                   type: integer
 *                   example: 1
 *                 created_at:
 *                   type: string
 *                   format: date-time
 *                   example: "2024-08-01T00:00:00.000Z"
 *                 updated_by:
 *                   type: integer
 *                   example: 2
 *                 updated_at:
 *                   type: string
 *                   format: date-time
 *                   example: "2024-09-01T00:00:00.000Z"
 *       400:
 *         description: Bad request
 */
contractorAppointmentRoutes.get(
  "/get-contractor-appointment/:id",
  verifyToken,
  sideBarPermissionCheck("view", 256, "contractor_appointments"),
  controllerWrapper(getContractorAppointment)
);

/**
 * @swagger
 * /api/contractorAppointment/delete-contractor-appointment/{id}:
 *   delete:
 *     summary: Delete a contractor appointment by ID
 *     tags: [Contractor Appointment]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The ID of the contractor appointment
 *     responses:
 *       200:
 *         description: Contractor appointment deleted successfully
 *       400:
 *         description: Bad request
 */
contractorAppointmentRoutes.delete(
  "/delete-contractor-appointment/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 256, "contractor_appointments"),
  controllerWrapper(deleteContractorAppointment)
);

/**
 * @swagger
 * /api/contractorAppointment/upload-contractor-appointment-signature:
 *   post:
 *     summary: Upload signature
 *     tags: [Contractor Appointment]
 */
contractorAppointmentRoutes.post(
  "/upload-contractor-appointment-signature",
  verifyToken,
  controllerWrapper(uploadSignature)
);

export default contractorAppointmentRoutes;
