// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  createUpdateAuditExecution,
  deleteAuditExecution,
  saveAuditRecord,
  viewAllAuditExecution,
  viewAllAuditRecord,
} from "../controller/auditExecutionController.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { validateRequest } from "../helper/general.js";
import {
  auditExecutionCreateUpdateValidationSchema,
  auditTemplateCreateUpdateValidationSchema,
} from "../validationsRecord/validationSchema.js";

/**
 * @swagger
 * tags:
 *   name: Audit Execution
 *   description: API operations for managing all Audit Execution
 */
const auditExecutionRoutes = Router();

/**
 * @swagger
 * /api/auditExecution/create-update-audit-execution:
 *   post:
 *     summary: Create or update an Audit Execution
 *     tags: [Audit Execution]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: string
 *               name:
 *                 type: string
 *               description:
 *                 type: string
 *               organization:
 *                 type: array
 *                 items:
 *                   type: string
 *           example: { id: "id", name: "name", description: "description", organization: [] }
 */
auditExecutionRoutes.post(
  "/create-update-audit-execution",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      156,
      "audit_execution"
    )(req, res, next);
  },
  validateRequest(auditExecutionCreateUpdateValidationSchema),
  controllerWrapper(createUpdateAuditExecution)
);

/**
 * @swagger
 * /api/auditExecution/get-audit-execution:
 *   get:
 *     summary: Get a list of all Audit Execution
 *     tags: [Audit Execution]
 */
auditExecutionRoutes.get(
  "/get-audit-execution",
  verifyToken,
  sideBarPermissionCheck("view", 156, "audit_execution"),
  controllerWrapper(viewAllAuditExecution)
);

/**
 * @swagger
 * /api/auditExecution/get-audit-execution/{id}:
 *   get:
 *     summary: Get a specific record of Audit Execution
 *     tags: [Audit Execution]
 *     parameters:
 *       - name: id
 *         in: path
 *         required: true
 *         description: ID of the audit execution to retrieve
 *         schema:
 *           type: string
 */
auditExecutionRoutes.get(
  "/get-audit-execution/:id",
  verifyToken,
  sideBarPermissionCheck("view", 156, "audit_execution"),
  controllerWrapper(viewAllAuditExecution)
);

/**
 * @swagger
 * /api/auditExecution/delete-audit-execution/{id}:
 *   delete:
 *     summary: Delete a specific record of Audit Execution
 *     tags: [Audit Execution]
 *     parameters:
 *       - name: id
 *         in: path
 *         required: true
 *         description: ID of the audit execution to delete
 *         schema:
 *           type: string
 */
auditExecutionRoutes.delete(
  "/delete-audit-execution/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 156, "audit_execution"),
  controllerWrapper(deleteAuditExecution)
);

/**
 * @swagger
 * /api/auditExecution/save-audit-record:
 *   post:
 *     summary: Save a specific audit record using audit template
 *     tags: [Audit Execution]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               name:
 *                 type: string
 *               description:
 *                 type: string
 *               organization:
 *                 type: array
 *                 items:
 *                   type: string
 *               department:
 *                 type: string
 *               questions:
 *                 type: array
 *                 items:
 *                   type: string
 *               audit_execution:
 *                 type: string
 *           example: { name: "name", description: "Description", organization: [], department: "department", questions: [], audit_execution: "audit_execution" }
 */
auditExecutionRoutes.post(
  "/save-audit-record",
  verifyToken,
  sideBarPermissionCheck("view", 156, "audit_execution"),
  // validateRequest(auditTemplateCreateUpdateValidationSchema),
  controllerWrapper(saveAuditRecord)
);

/**
 * @swagger
 * /api/auditExecution/get-audit-record:
 *   get:
 *     summary: Get a list of all Audit Records
 *     tags: [Audit Execution]
 */
auditExecutionRoutes.get(
  "/get-audit-record",
  verifyToken,
  sideBarPermissionCheck("view", 155, "audit_execution"),
  controllerWrapper(viewAllAuditRecord)
);

/**
 * @swagger
 * /api/auditExecution/get-audit-record/{id}:
 *   get:
 *     summary: Get a specific record of Audit Record
 *     tags: [Audit Execution]
 *     parameters:
 *       - name: id
 *         in: path
 *         required: true
 *         description: ID of the audit record to retrieve
 *         schema:
 *           type: string
 */
auditExecutionRoutes.get(
  "/get-audit-record/:id",
  verifyToken,
  sideBarPermissionCheck("view", 155, "audit_execution"),
  controllerWrapper(viewAllAuditRecord)
);

export default auditExecutionRoutes;
