// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { validateRequest } from "../helper/general.js";
import {
  deleteSOP,
  SOPCreateUpdate,
  viewSOP,
  sopComment,
  viewSopApprovalWorkFlow,
  getSopHistory,
} from "../controller/sopController.js";

/**
 * @swagger
 * tags:
 *   name: SOP
 *   description: API Operations For Managing SOP
 */
const SOPRoutes = Router();

/**
 * @swagger
 * /api/SOP/create-update-sop:
 *   post:
 *     summary: Create or Update a SOP
 *     tags: [SOP]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               amendment:
 *                 type: boolean
 *                 example: false
 *               new_sop:
 *                 type: boolean
 *                 example: true
 *               sop_id:
 *                 type: string
 *                 example: SOP-001
 *               sop_name:
 *                 type: string
 *                 example: "Fire Safety Protocol"
 *               sop_number:
 *                 type: string
 *                 example: "SOP123"
 *               sop_classification:
 *                 type: string
 *                 enum: ["Unclassified", "Confidential", "Restricted", "Private"]
 *                 example: "Confidential"
 *               sop_process:
 *                 type: string
 *                 example: "Fire safety measures"
 *               sop_category:
 *                 type: integer
 *                 example: 3
 *               version_number:
 *                 type: string
 *                 example: "1.0"
 *               sop_effective_date:
 *                 type: string
 *                 format: date
 *                 example: "2024-01-01"
 *               last_review_date:
 *                 type: string
 *                 format: date
 *                 example: "2024-06-30"
 *               next_review_date:
 *                 type: string
 *                 format: date
 *                 example: "2025-06-30"
 *               sop_status:
 *                 type: string
 *                 enum: ["Draft", "Complete"]
 *                 example: "Draft"
 *               sop_purpose:
 *                 type: string
 *                 example: "Ensure fire safety in all areas"
 *               sop_scope:
 *                 type: string
 *                 example: "Applicable across all departments"
 *               sop_statement:
 *                 type: string
 *                 example: "This SOP outlines fire safety protocols"
 *               sop_description:
 *                 type: string
 *                 example: "Detailed fire safety procedures"
 *               sop_roles_responsibilities:
 *                 type: object
 *                 example:
 *                   role: "Fire Safety Officer"
 *                   responsibility: "Ensure all fire safety equipment is operational"
 *               sop_revision_history:
 *                 type: array
 *                 items:
 *                   type: object
 *                   properties:
 *                     version_number:
 *                       type: string
 *                       example: "1.0"
 *                     effective_date:
 *                       type: string
 *                       format: date
 *                       example: "2024-01-01"
 *                     details_of_addition:
 *                       type: string
 *                       example: "Initial release"
 *               file_name:
 *                 type: string
 *                 example: "fire_safety_sop.pdf"
 *               file_classification:
 *                 type: string
 *                 example: "Confidential"
 *               notify_to:
 *                 type: array
 *                 items:
 *                   type: object
 *                   properties:
 *                     employee_name:
 *                       type: string
 *                       example: "John Doe"
 *                     role:
 *                       type: string
 *                       example: "Safety Officer"
 *                     department:
 *                       type: string
 *                       example: "Safety Department"
 *               sop_author:
 *                 type: integer
 *                 example: 101
 *               sop_reviewer:
 *                 type: array
 *                 items:
 *                   type: object
 *                   properties:
 *                     employee_name:
 *                       type: string
 *                       example: "Jane Smith"
 *                     role:
 *                       type: string
 *                       example: "Senior Reviewer"
 *                     department:
 *                       type: string
 *                       example: "Compliance"
 *               sop_approver:
 *                 type: integer
 *                 example: 102
 *               department:
 *                 type: integer
 *                 example: 5
 *               organization:
 *                 type: integer
 *                 example: 14
 */

SOPRoutes.post(
  "/create-update-sop",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(req.body.id ? "edit" : "create", 307, "sops")(req, res, next);
  },
  controllerWrapper(SOPCreateUpdate)
);

/**
 * @swagger
 * /api/SOP/get-sop:
 *   get:
 *     summary: Get list of all SOPs
 *     tags: [SOP]
 */
SOPRoutes.get("/get-sop", verifyToken, sideBarPermissionCheck("view", 307, "sops"), controllerWrapper(viewSOP));

/**
 * @swagger
 * /api/SOP/get-sop/{id}:
 *   get:
 *     summary: Get A Specific SOP record
 *     tags: [SOP]
 */
SOPRoutes.get("/get-sop/:id", verifyToken, sideBarPermissionCheck("view", 307, "sops"), controllerWrapper(viewSOP));

/**
 * @swagger
 * /api/SOP/delete-sop/{id}:
 *   delete:
 *     summary: Delete a specific record of SOP
 *     tags: [SOP]
 */
SOPRoutes.delete(
  "/delete-sop/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 307, "sops"),
  controllerWrapper(deleteSOP)
);

SOPRoutes.post("/sop-comment", verifyToken, sideBarPermissionCheck("view", 307, "sops"), controllerWrapper(sopComment));

SOPRoutes.get(
  "/get-sop-approval-workflow",
  verifyToken,
  sideBarPermissionCheck("view", 307, "sops"),
  controllerWrapper(viewSopApprovalWorkFlow)
);

SOPRoutes.get(
  "/get-sop-approval-workflow/:id",
  verifyToken,
  sideBarPermissionCheck("view", 307, "sops"),
  controllerWrapper(viewSopApprovalWorkFlow)
);
SOPRoutes.get("/get-sop-history/:id", verifyToken, controllerWrapper(getSopHistory));

export default SOPRoutes;
