// Import necessary modules
import express from "express";
import path from "path";
import cors from "cors";
import fileUpload from "express-fileupload";
import http from "http";
import dotenv from "dotenv";
import swaggerUi from "swagger-ui-express";
import swaggerSpec from "./swaggerConfig.js";
import db from "./db-config.js";
import { verifyToken } from "./helper/tokenVerify.js";
import routes from "./routes/routes.js";
import {
  decodeAndParseFields,
  deleteAllRecord,
  getFinancialYear,
  getInitials,
  storeError,
  uniqueIdGenerator,
  updateOrganizationColumnFormat,
} from "./helper/general.js";
import { controllerWrapper, sendResponse } from "./helper/wrapper.js";
import { getShortUrl } from "./controller/surveyDesignController.js";
import chalk from "chalk";
import file from "./helper/importModel.js";
import { setupClientCrashHandler } from "./util/clientAppCrashHandler.js";
import cronJob from "./jobs/cronJobs.js";
import { createTriggersForAllTables } from "./seeders/processTriggers.js";
// import rateLimit from "express-rate-limit";

dotenv.config();
const app = express();

cronJob();

// const limiter = rateLimit({
//   windowMs: 1 * 60 * 1000, // 1 minutes in milliseconds
//   max: 100, // Limit each IP to 50 requests per 1 min
//   message: "Too many requests from this IP, please try again after 15 minutes.",
// });

// app.use(limiter);

const PORT = process.env.PORT || 8007;
const server = http.createServer(app);

// initializeSocket(server);

app.use(cors());
app.use(fileUpload());
app.use(express.urlencoded({ extended: true }));
app.use(express.json({ limit: "2000mb" }));
app.use(express.static(path.resolve("./public")));
app.use(cors({ origin: "*", allowedHeaders: "Authorization" }));
app.use("/api-docs", swaggerUi.serve, swaggerUi.setup(swaggerSpec));

// Initialize crash handler at the start of client service
setupClientCrashHandler(process.env.SUPER_ADMIN_BACKEND_URL);

app.get("/docs/swagger.json", (req, res) => {
  res.json(swaggerSpec);
});

app.use((req, res, next) => {
  console.log(req.path)
  return next()
})

/**Routes */

app.use("/api", routes);

app.get("/si/:shortUrl", controllerWrapper(getShortUrl));

app.get("/getJson", async (req, res) => {
  const permission = [];
  let [organizationList] = await db.query(
    `SELECT * FROM organization where deleted = 0`
  );

  let organization = organizationList.map((item) => item.id);
  let permissions = {};
  organization.forEach((item) => {
    permissions[item] = { view: true, edit: true, delete: true, create: true };
  });
  for (let i = 1; i <= 500; i++) {
    permission.push({
      sidebarId: i,
      permissions: permissions,
    });
  }

  const [result] = await db.query(
    `UPDATE users SET permission='${JSON.stringify(
      permission
    )}', organization = '${JSON.stringify(organization)}' WHERE id='1' `
  );
  return res.status(200).json({
    status: true,
    message: "ho gya",
  });
});
/**Function to fetch all country data */
app.get("/api/country", async (req, res) => {
  try {
    const countryDataQuery = `SELECT id,name FROM country order by name asc`;
    const [countryData] = await db.query(countryDataQuery);
    return res.status(200).json({
      status: true,
      data: countryData,
    });
  } catch (err) {
    storeError(err);
    res.status(500).json({
      status: false,
      message: "An error occurred",
      error: err.message,
    });
  }
});

/**Function to fetch all state data according to country */
app.get("/api/state/:country_id", async (req, res) => {
  try {
    const { country_id } = req.params;
    const stateDataQuery = `SELECT id,name FROM states WHERE country_id = ${country_id} order by name asc`;
    const [stateData] = await db.query(stateDataQuery);
    return res.status(200).json({
      status: true,
      data: stateData,
    });
  } catch (err) {
    storeError(err);
    res.status(500).json({
      status: false,
      message: err.message,
    });
  }
});

app.get("/api/city/:state_id", async (req, res) => {
  try {
    const { state_id } = req.params;
    const cityDataQuery = `SELECT id,name FROM cities WHERE state_id = ${state_id} order by name asc`;
    const [cityData] = await db.query(cityDataQuery);
    return res.status(200).json({
      status: true,
      data: cityData,
    });
  } catch (error) {
    storeError(error);
    res.status(500).json({
      status: false,
      message: "An error occurred",
      error: error.message,
    });
  }
});

app.post("/api/delete-multiple-record", verifyToken, deleteAllRecord);

/**Function to fetch all city data according to state */
/**Function to fetch all city data according to region */
app.get("/api/city/region/:region_id", async (req, res) => {
  try {
    const { region_id } = req.params;
    const cityDataQuery = `SELECT id,name FROM cities WHERE region_id = ${region_id} order by name asc`;
    const [cityData] = await db.query(cityDataQuery);
    return res.status(200).json({
      status: true,
      data: cityData,
    });
  } catch (error) {
    storeError(error);
    res.status(500).json({
      status: false,
      message: "An error occurred",
      error: error.message,
    });
  }
});

/**Function to fetch all region data according to state */
app.get("/api/getregion/:state_id", async (req, res) => {
  try {
    const { state_id } = req.params;
    // console.log(state_id)
    const cityDataQuery = `SELECT id,name FROM region WHERE state_id = ${state_id} order by name asc`;
    const [cityData] = await db.query(cityDataQuery);
    return res.status(200).json({
      status: true,
      data: cityData,
    });
  } catch (error) {
    storeError(error);
    res.status(500).json({
      status: false,
      message: "An error occurred",
      error: error.message,
    });
  }
});

/**Function to get side bar records*/
app.get("/api/sidebarPermission/sidebar", verifyToken, async (req, res) => {
  try {
    let [permissionsData] = await db.query(
      `SELECT permission, role FROM users WHERE id = ${req.user.sessionid}`
    );

    if (permissionsData[0].permission == null) {
      [permissionsData] = await db.query(
        `SELECT permission FROM roles WHERE id = ${permissionsData[0].role}`
      );
    }

    const permissions = permissionsData[0].permission
      ? JSON.parse(permissionsData[0].permission)
      : [];
    const modules = await getModules(permissions, 0);
    return res.status(200).json({
      status: true,
      data: modules,
    });
  } catch (error) {
    storeError(error);
    res.status(500).json({
      status: false,
      message: "An error occurred",
      error: error.message,
    });
  }
});

app.use((err, req, res, next) => {
  // Log the error and the IP address
  console.error(chalk.redBright(`Error: ${err.message}`));
  console.error(chalk.redBright(`Request IP: ${req.ip}`));

  // Set the status code and send the error response
  res.status(err.status || 500);
  res.json({
    message: err.message,
    // Optionally include stack trace (not recommended in production)
    stack: process.env.NODE_ENV === "production" ? null : err.stack,
    ip: req.ip,
  });
});

app.get(
  "/api/action",
  controllerWrapper(async (req, res) => {
    const { organization, sub_module_id } = req.query;
    const [tableName] = await db.query(
      `SELECT table_name FROM sidebar WHERE id = ?`,
      [sub_module_id]
    );
    if (!tableName[0]?.table_name) {
      return sendResponse(res, 200, []);
    }
    let query = `SELECT * FROM ${tableName[0].table_name} WHERE organization = ? AND deleted = 0 ORDER BY id DESC`;
    if (tableName[0].table_name === "permit_license_compliance") {
      query = `SELECT permit_license_compliance.*, permit_license.name as name FROM permit_license_compliance LEFT JOIN permit_license ON permit_license.id = permit_license_compliance.name WHERE permit_license.organization = ? AND permit_license.deleted = 0 ORDER BY id DESC`;
    }
    let [result] = await db.query(query, [organization]);
    result = await decodeAndParseFields(result);

    return sendResponse(res, 200, result);
  })
);

async function getModules(permissions, parentId) {
  const [modules] = await db.query(
    `SELECT id, title, icon, path ,type FROM sidebar WHERE parent_id = ${parentId} AND deleted = 0 order by sequence asc`
  );
  const result = [];
  for (const moduleData of modules) {
    // add backend url to icon path
    if (moduleData.icon) {
      moduleData.icon = `${process.env.HARMONY_BACKEND_URL}/${moduleData.icon}`;
    }
    const modulePermission = permissions.find(
      (permission) => permission.sidebarId === moduleData.id
    );
    // console.log("modulePermission: ", modulePermission);
    if (modulePermission) {
      moduleData.permissions = modulePermission.permissions;
    } else {
      moduleData.permissions = {};
    }

    moduleData.submodules = await getModules(permissions, moduleData.id);
    // Check if there is any true permission in the current module or its submodules
    if (hasTruePermission(moduleData)) {
      result.push(moduleData);
    }
  }
  return result;
}

function hasTruePermission(moduleData) {
  // Check current module permissions
  for (const key in moduleData.permissions) {
    if (Object.values(moduleData.permissions[key]).includes(true)) {
      return true;
    }
  }

  // Recursively check submodules permissions
  for (const submodule of moduleData.submodules) {
    if (hasTruePermission(submodule)) {
      return true;
    }
  }

  return false;
}
export { getModules };

// sideBarPermission.get("/sidebar", verifyToken, sideBarRecord);

// app.get("/:folder/:filename", controllerWrapper((req, res) => {
//   const fileName = req.params.filename;
//   const folder = req.params.folder
//   const filePath = path.join(process.cwd(), "public", folder, fileName);
//  return res.sendFile(filePath);
// }));

const removeAllForeignKeyConstraints = async () => {
  // Connect to the database
  try {
    // Step 1: Find all foreign key constraints
    const query = `
      SELECT
        TABLE_NAME,
        CONSTRAINT_NAME
      FROM
        INFORMATION_SCHEMA.KEY_COLUMN_USAGE
      WHERE
        TABLE_SCHEMA = 'harmony_help_default'
        AND REFERENCED_TABLE_NAME IS NOT NULL;
    `;

    const [results] = await db.query(query);

    if (results.length === 0) {
      console.log("No foreign key constraints found.");
      return;
    }

    // Step 2: Iterate over the results and drop each foreign key constraint
    for (const row of results) {
      const alterQuery = `ALTER TABLE ${row.TABLE_NAME} DROP FOREIGN KEY ${row.CONSTRAINT_NAME};`;

      try {
        await db.query(alterQuery);
        console.log(
          `Removed constraint ${row.CONSTRAINT_NAME} from table ${row.TABLE_NAME}.`
        );
      } catch (err) {
        console.error(
          `Error removing constraint ${row.CONSTRAINT_NAME} from table ${row.TABLE_NAME}:`,
          err
        );
      }
    }
  } catch (err) {
    console.error("Error querying the database:", err);
  }
};

//  removeAllForeignKeyConstraints();

// await removeAllForeignKeyConstraints();
server.listen(PORT, () => {
  console.log(chalk.bgCyan(`This server is running on port ${PORT}`));
});

app.get("/sidebar-list", async (req, res) => {
  const [results] = await db.query(
    `SELECT id, title, icon, path ,type, table_name, parent_id, deleted, sequence FROM harmony_help3.sidebar WHERE  deleted = 0 order by sequence asc`
  );
  return res.status(200).json({
    status: true,
    data: results,
  });
});

app.get("/api/get-type-of-property", async (req, res) => {
  try {
    const data = [
      { id: 1, label: "Residential Properties", parentId: null },
      { id: 2, label: "Single-Family Homes", parentId: 1 },
      { id: 3, label: "Multi-Family Homes", parentId: 1 },
      { id: 4, label: "Condominiums", parentId: 1 },
      { id: 5, label: "Townhouses", parentId: 1 },
      { id: 6, label: "Duplexes", parentId: 1 },
      { id: 7, label: "Apartments", parentId: 1 },
      { id: 8, label: "Mobile Homes", parentId: 1 },
      { id: 9, label: "Vacation Homes", parentId: 1 },
      { id: 10, label: "Co-Op Housing", parentId: 1 },
      { id: 11, label: "Commercial Properties", parentId: null },
      { id: 12, label: "Office Buildings", parentId: 11 },
      { id: 13, label: "Retail Spaces", parentId: 11 },
      { id: 14, label: "Shopping Centers", parentId: 11 },
      { id: 15, label: "Malls", parentId: 11 },
      { id: 16, label: "Hotels", parentId: 11 },
      { id: 17, label: "Industrial Buildings", parentId: 11 },
      { id: 18, label: "Warehouses", parentId: 11 },
      { id: 19, label: "Mixed-Use Buildings", parentId: 11 },
      { id: 20, label: "Medical Centers", parentId: 11 },
      { id: 21, label: "Restaurants", parentId: 11 },
      { id: 22, label: "Auto Dealerships", parentId: 11 },
      { id: 23, label: "Gas Stations", parentId: 11 },
      { id: 24, label: "Parking Lots", parentId: 11 },
      { id: 25, label: "Industrial Properties", parentId: null },
      { id: 26, label: "Manufacturing Plants", parentId: 25 },
      { id: 27, label: "Factories", parentId: 25 },
      { id: 28, label: "Distribution Centers", parentId: 25 },
      {
        id: 29,
        label: "Research and Development (R&D) Facilities",
        parentId: 25,
      },
      { id: 30, label: "Cold Storage", parentId: 25 },
      { id: 31, label: "Data Centers", parentId: 25 },
      { id: 32, label: "Logistics Parks", parentId: 25 },
      { id: 33, label: "Agricultural Properties", parentId: null },
      { id: 34, label: "Farms", parentId: 33 },
      { id: 35, label: "Ranches", parentId: 33 },
      { id: 36, label: "Orchards", parentId: 33 },
      { id: 37, label: "Vineyards", parentId: 33 },
      { id: 38, label: "Plantations", parentId: 33 },
      { id: 39, label: "Dairy Farms", parentId: 33 },
      { id: 40, label: "Poultry Farms", parentId: 33 },
      { id: 41, label: "Special Purpose Properties", parentId: null },
      { id: 42, label: "Schools", parentId: 41 },
      { id: 43, label: "Churches", parentId: 41 },
      { id: 44, label: "Cemeteries", parentId: 41 },
      { id: 45, label: "Hospitals", parentId: 41 },
      { id: 46, label: "Government Buildings", parentId: 41 },
      { id: 47, label: "Airports", parentId: 41 },
      { id: 48, label: "Stadiums", parentId: 41 },
      { id: 49, label: "Theaters", parentId: 41 },
      { id: 50, label: "Amusement Parks", parentId: 41 },
      { id: 51, label: "Golf Courses", parentId: 41 },
      { id: 52, label: "Marinas", parentId: 41 },
      { id: 53, label: "Correctional Facilities", parentId: 41 },
      { id: 54, label: "Vacant Land", parentId: null },
      { id: 55, label: "Raw Land", parentId: 54 },
      { id: 56, label: "Infill Land", parentId: 54 },
      { id: 57, label: "Agricultural Land", parentId: 54 },
      { id: 58, label: "Urban Land", parentId: 54 },
      { id: 59, label: "Suburban Land", parentId: 54 },
      { id: 60, label: "Forest Land", parentId: 54 },
      { id: 61, label: "Recreational Land", parentId: 54 },
      { id: 62, label: "Mixed-Use Properties", parentId: null },
      { id: 63, label: "Residential-Commercial Combos", parentId: 62 },
      { id: 64, label: "Live-Work Units", parentId: 62 },
      { id: 65, label: "Retail-Office Complexes", parentId: 62 },
      { id: 66, label: "Transit-Oriented Developments", parentId: 62 },
      { id: 67, label: "Multi-Use High-Rises", parentId: 62 },
      { id: 68, label: "Recreational Properties", parentId: null },
      { id: 69, label: "Resorts", parentId: 68 },
      { id: 70, label: "Campsites", parentId: 68 },
      { id: 71, label: "Vacation Rentals", parentId: 68 },
      { id: 72, label: "Hunting Grounds", parentId: 68 },
      { id: 73, label: "Ski Resorts", parentId: 68 },
      { id: 74, label: "Private Islands", parentId: 68 },
      { id: 75, label: "Beaches", parentId: 68 },
      { id: 76, label: "Institutional Properties", parentId: null },
      { id: 77, label: "Museums", parentId: 76 },
      { id: 78, label: "Libraries", parentId: 76 },
      { id: 79, label: "Universities", parentId: 76 },
      { id: 80, label: "Military Bases", parentId: 76 },
      { id: 81, label: "Research Institutions", parentId: 76 },
    ];
    return res.status(200).json({
      status: true,
      data: data,
    });
  } catch (error) {
    storeError(error);
    res.status(500).json({
      status: false,
      message: "An error occurred",
      error: error.message,
    });
  }
});

/// for updating the organization key from integer to text
const updatePart = [
  "certificates",
  "labels",
  "waste_type",
  "first_aid_box",
  "equipment",
  "responsibility",
  "skills",
  "gender",
  "license",
  "appointment_type",
  "role_hierarchy",
  "highest_qualification",
  "disability",
  "employee_type",
  "language",
  "reason_for_inactive",
  "race",
  "relation",
  "document_classification",
  "parent_objective",
  "currency",
  "contractor_forms",
  "type_of_service",
  "banks",
  "role_on_the_project",
  "location",
  "financial_year",
  "category",
  "business_processes",
  "collection_site",
  "disposal_site",
  "communication_method",
  "competencies_assessed",
  "file_classification",
  "main_process",
  "waste_disposal_method",
  "collection_site",
  "insurance_policy_impact",
  "incident_role",
  "quality_incident_type",
  "defect_type",
  "defect_detection_method",
  "waste_description",
  "shift_information",
  "incident_ppe",
  "name_of_standard",
  "material_classification",
  "asset_type",
  "severity",
  "regulator",
  "engagement_levels",
  "feedback_context",
  "affected_area",
  "critical_system_affected",
  "document_type",
  "sop_category",
  "training_disclaimer",
  "crisis_type",
  "occupational_disease",
  "issuing_authority",
  "permit_license",
  "incident_category",
  "engagement_method",
  "incident_location",
  "focus_area",
  "measurement_metric",
  "name_of_standard",
  "occupational_disease",
  "audit_template",
  "audit_type",
  "assessor_method",
  "best_practice",
  "gender",
  "purpose_of_training",
  "security_incident_type",
  "crisis_type",
  "processes",
  "audit_category",
];
// for (let update of updatePart) {
//   console.log("update", update);
//   await updateOrganizationColumnFormat(update)
// }

const updateNullUniqueIds = async (tableName, key = "unique_id") => {
  try {
    // Get all records that need unique IDs
    const getNullRecordsQuery = `
      SELECT 
        s.id,
        s.organization AS organizationId,
        s.created_at,
        COALESCE(o.name, '') AS organizationName  -- Ensure organization name is not null
      FROM ${tableName} s
      LEFT JOIN organization o ON s.organization = o.id
      ORDER BY s.created_at ASC`;
    //WHERE s.${key} IS NULL

    const [nullRecords] = await db.query(getNullRecordsQuery);
    console.log("Found records to update:", nullRecords.length);

    for (const record of nullRecords) {
      try {
        // Convert organizationId from JSON string to integer (if needed)
        let organizationId = record.organizationId;
        if (typeof organizationId === "string") {
          try {
            const parsedId = JSON.parse(organizationId);
            organizationId = Array.isArray(parsedId) ? parsedId[0] : parsedId; // Extract first ID if array
          } catch (err) {
            console.error(
              `Failed to parse organizationId for record ${record.id}:`,
              err
            );
          }
        }

        console.log(`Processed organizationId: ${organizationId}`);

        // Get organization initials
        let orgInitials = organizationId
          ? await getInitials(organizationId, "organization", "name")
          : "UNK"; // Default to 'UNK' if no valid organization

        console.log(`Organization Initials: ${orgInitials}`);

        // **Dynamically generate module prefix (first letter of table name)**
        const modulePrefix = tableName.slice(0, 3).toUpperCase();
        // Example: 'skills' → 'S', 'projects' → 'P'

        console.log(`Module Prefix: ${modulePrefix}`);

        // Get financial year
        const year = getFinancialYear(new Date());

        // Get current max sequence number
        const maxIdQuery = `
          SELECT ${key} 
          FROM ${tableName} 
          WHERE ${key} LIKE ? 
          ORDER BY LENGTH(${key}) DESC, ${key} DESC 
          LIMIT 1`;

        const likePattern = `${orgInitials}/${modulePrefix}/${year}/%`;
        const [maxId] = await db.query(maxIdQuery, [likePattern]);

        let sequence = 1;
        if (maxId.length > 0 && maxId[0][key]) {
          const currentSequence = maxId[0][key].split("/").pop();
          sequence = isNaN(currentSequence) ? 1 : parseInt(currentSequence) + 1;
        }

        // Generate new unique ID
        const newUniqueId = `${orgInitials}/${modulePrefix}/${year}/${sequence
          .toString()
          .padStart(4, "0")}`;

        // Update the record
        await db.query(
          `UPDATE ${tableName} 
           SET ${key} = ? 
           WHERE id = ?`,
          [newUniqueId, record.id]
        );

        console.log(`Updated record ${record.id} with ID: ${newUniqueId}`);
      } catch (recordError) {
        console.error(`Error processing record ${record.id}:`, recordError);
        continue;
      }
    }

    return {
      status: true,
      message: `Updated ${nullRecords.length} records with new unique IDs`,
      updatedCount: nullRecords.length,
    };
  } catch (error) {
    console.error("Error updating unique IDs:", error);
    return {
      status: false,
      message: error.message,
    };
  }
};

// for updating unique id from null to not null for all static data for default data base
let staticDataArrayTable = [
  // "illness_type",
  // "incident_location", // -- not in use
  // "incident_ppe", // -- not in use
  // "issuing_authority",
  // "license",
  // "location",
  // "measurement_metric",
  // "parent_objective",
  // "permit_license",
  "ppe_type",
  // "skills",
  // "document_type",
  // "focus_area",
  // "business_processes",
  // "quality_incident_type"
];
// for(let table of staticDataArrayTable){
//    await updateNullUniqueIds(table)
// }

// creating create and update triggers for any update for all table
// createTriggersForAllTables();
