import chalk from "chalk";
import db from "../db-config.js";
import fs from "fs";
import path from "path";
import { createQueryBuilder } from "./queryBuilder.js";
import Repository from "../sequelize/RepositorySchema.js";
import DocumentCreation from "../sequelize/DocumentCreationSchema.js";
import { fileURLToPath } from "url";
import { auditLogFunction } from "./general.js";
const __filename = fileURLToPath(import.meta.url);
let __dirname = path.dirname(__filename);
let stack = [];

// Recursive function to get parent titles
const getParent = async (id) => {
  try {
    const [parentId] = await db.query(`SELECT * FROM sidebar WHERE id = ?`, [id]);

    //* Check if a parentId exists and process the result
    if (parentId && parentId.length > 0) {
      const title = parentId[0]?.title;

      if (title) {
        console.log(chalk.gray(`Adding to stack: ${title}`));
        // stack.push(title); // Push the parent title to the stack

        stack.push({
          title,
          sidebarId: parentId[0]?.id,
        }); // Push current data to the stack

        //! If parent_id is not 0, keep going up the chain recursively
        if (parentId[0]?.parent_id !== 0) {
          await getParent(parentId[0]?.parent_id);
        }
      }
    }
  } catch (error) {
    console.error(chalk.red("Error in getParent function:", error));
  }
};

export const uploadToDDRM = async (sidebarId, file, req) => {
  try {
    console.log(chalk.blue(`Fetching data for sidebarId: ${sidebarId}`));

    if (!sidebarId || !file) {
      console.log(chalk.red(`Missing sidebarId or file sidebarId: ${sidebarId} and file: ${file}`));
      return null;
    }

    const [data] = await db.query(`SELECT * FROM sidebar WHERE id = ?`, [sidebarId]);

    if (data && data.length > 0) {
      const currentTitle = data[0]?.title;
      console.log(chalk.blue(`Starting with title: ${currentTitle}`));

      // ! First empty the stack for continuous recursion call
      stack = [];
      stack.push({
        title: currentTitle,
        sidebarId: sidebarId,
      }); // Push current data to the stack

      //! If there's a parent_id, start recursive retrieval
      if (data[0]?.parent_id !== 0) {
        await getParent(data[0]?.parent_id); // Await the recursive function
      }

      //* After recursion completes, log the full stack and create folders
      console.log(chalk.red(`Full stack: ${JSON.stringify(stack)}`));
      let { currentPath: uploadedPath, relativePath, parentId } = await createFoldersFromStack(stack.reverse(), req);
      // console.log("relativePath: ", relativePath);

      console.log(chalk.green(`Uploaded to: ${uploadedPath}`));
      //* Move the file to the final destination
      const fileName = `${Date.now()}-${file.name}`;
      uploadedPath = path.join(uploadedPath, fileName);
      relativePath = path.join(relativePath, fileName);
      // let dbPath = uploadedPath.split("public")[1];
      console.log(chalk.bgRed("parentId is:", parentId));

      console.log(chalk.green(`Moving to: ${uploadedPath}`));

      await file.mv(uploadedPath);
      // console.log('file.size: ', file.size);

      const document_name = file?.name?.split(".")[0] || "";
      // TODO - First create the document then create the repository
      const documentData = {
        upload_document: relativePath,
        organization: req?.body?.organization,
        created_by: req.user.sessionid,
        document_name: document_name,
      };

      const { query: DocumentCreationQuery, values: DocumentCreationValues } = createQueryBuilder(
        DocumentCreation,
        documentData
      );
      const [result] = await db.query(DocumentCreationQuery, DocumentCreationValues);

      const [childRepositoryData] = await db.query(`SELECT * FROM repository WHERE sidebar_id = ?`, [sidebarId]);
      const mainParent = childRepositoryData[0]?.id;

      const { query, values } = createQueryBuilder(Repository, {
        // name: file.name,
        // url: finalDestination,
        // parent_id: parentId,
        // organization: req?.body?.organization,
        // type: "file",
        // created_by: req.user.sessionid,
        name: result?.document_name || file.name,
        url: relativePath,
        parent_id: mainParent,
        organization: req?.body?.organization,
        type: "file",
        size: file.size,
        // document_creation_id: id ? id : result.insertId,
        // created_by: id ? result?.created_by : req.user.sessionid,
        // updated_by: id ? req.user.sessionid : null,
        document_creation_id: result.insertId,
        created_by: req.user.sessionid,
      });
      const [createdData] = await db.query(query, values);
      // add log here

      const auditData = {
        organization: req?.body?.organization,
        department: req?.body?.department,
        repository_id: createdData.insertId,
        created_by: req.user.sessionid,
        document_name: document_name,
        action_type: "upload",
        document_status: "In Progress",
      };
      await auditLogFunction(auditData);
      return createdData.insertId;
      // const { ddrmId, ddrmName, ddrmPath } = await getDDRMData(createdData.insertId);
      // return  { ddrmId, ddrmName, ddrmPath }
    } else {
      console.log(chalk.red("No data found for sidebarId:", sidebarId));
    }
  } catch (error) {
    console.error(chalk.red("Error in uploadToDDRM function:", error));
    return null;
  }
};

export const newUploadToDDRM = async (file, sidebarId, req) => {
  try {
    if (!sidebarId || !file) {
      console.log(chalk.red(`Missing sidebarId or file sidebarId: ${sidebarId} and file: ${file}`));
      return null;
    }

    const [data] = await db.query(`SELECT * FROM sidebar WHERE id = ?`, [sidebarId]);
    // console.log("data: ", data);

    if (data && data.length > 0) {
      const currentTitle = data[0]?.title;
      console.log(chalk.blue(`Starting with title: ${currentTitle}`));

      // ! First empty the stack for continuous recursion call
      stack = [];
      stack.push({
        title: currentTitle,
        sidebarId: sidebarId,
      }); // Push current data to the stack

      //! If there's a parent_id, start recursive retrieval
      if (data[0]?.parent_id !== 0) {
        await getParent(data[0]?.parent_id); // Await the recursive function
      }

      //* After recursion completes, log the full stack and create folders
      console.log(chalk.red(`Full stack: ${JSON.stringify(stack)}`));
      let { currentPath: uploadedPath, relativePath } = await createFoldersFromStack(stack.reverse(), req);
      const [childRepositoryData] = await db.query(`SELECT * FROM repository WHERE sidebar_id = ?`, [sidebarId]);
      const parentId = childRepositoryData[0]?.id;
      return { uploadedPath, relativePath, parentId };
    }
  } catch (error) {
    console.error("Error creating folder structure:", error);
    throw error;
  }
};

const createFoldersFromStack = async (stack, req) => {
  try {
    // __dirname = __dirname.startsWith("/") ? __dirname.slice(1) : __dirname;
    let currentPath = path.join(process.cwd(), "public", "root", "ddrm");
    console.log(chalk.bgRedBright(`Creating folders under: ${currentPath}`));

    const [parentData] = await db.query(
      `SELECT * FROM repository WHERE name = 'ddrm' AND deleted = 0 AND type = 'folder' AND organization = '${req?.body?.organization}'`
    );
    // console.log("parentData: ", parentData);
    let parentId = parentData[0]?.id || 0;
    let relativePath = path.join("root", "ddrm");
    // Iterate over the stack to create nested folders in LIFO order
    for (let folder of stack) {
      const sidebarId = folder?.sidebarId;
      folder = folder?.title;
      currentPath = path.join(currentPath, folder);
      relativePath = path.join(relativePath, folder);

      // Log the path that is about to create (for debugging)

      try {
        await fs.promises.access(currentPath, fs.constants.F_OK);
        const [folderExist] = await db.query(`SELECT * FROM repository WHERE url = ? AND organization = ?`, [
          relativePath,
          req?.body?.organization,
        ]);
        if (!folderExist[0]) {
          parentId = await createFolder(folder, sidebarId, currentPath, relativePath, parentId, req); // Create folder and get new parentId
        }
        console.log(chalk.gray(`Folder already exists: ${currentPath}`));
      } catch (error) {
        parentId = await createFolder(folder, sidebarId, currentPath, relativePath, parentId, req); // Create folder and get new parentId
        console.log(chalk.bgGray(`Folder created at: ${currentPath}`));
        //* Create the folder (if it doesn't exist) recursively
      }
    }
    return { currentPath, relativePath, parentId };
  } catch (error) {
    console.error("Error creating folder structure:", error);
  }
};

const createFolder = async (folder, sidebarId, currentPath, relativePath, parentId, req) => {
  const { query, values } = createQueryBuilder(Repository, {
    name: folder,
    url: relativePath,
    parent_id: parentId,
    organization: req?.body?.organization,
    type: "folder",
    created_by: req.user.sessionid,
    sidebar_id: sidebarId,
    is_locked: 1,
  });

  try {
    console.log(chalk.gray(`sidebar for folder: ${sidebarId}`));
    //* Create the folder (if it doesn't exist) recursively
    await fs.promises.mkdir(currentPath, { recursive: true });
    console.log("currentPath: ", currentPath);

    //! Insert folder into the database and return the result
    const [createdData] = await db.query(query, values);
    console.log(chalk.bgRedBright(`Inserted to database:`, createdData));
    return createdData.insertId;
  } catch (error) {
    console.error(chalk.red(`Error creating folder ${folder} at: ${currentPath}`), error);
    throw error;
  }
};

export const getDDRMData = async (id) => {
  try {
    const [data] = await db.query(`SELECT * FROM repository WHERE id = ?`, [id]);
    console.log(data);
    return {
      ddrmId: data[0]?.id,
      ddrmName: data[0]?.name,
      ddrmPath: data[0]?.url,
    };
  } catch (error) {
    console.log(`Error getting the repository details: ${error}`);
  }
};
