import asyncHandler from "express-async-handler";
import db from "../db-config.js";
import {
  countQueryCondition,
  deleteRecord,
  getOrganizationAccordingToDepartment,
  insertActivityLog,
  makeJoins,
  whereCondition,
  encodeSingle_statement,
  decodeSingle_statement,
  decodeTheEditorContent,
  encodeTheEditorContent,
  updateQueryBuilder,
  createQueryBuilder,
  searchConditionRecord,
  decodeAndParseFields,
  riskRegisterTemplateHelper,
} from "../helper/general.js";
import TacticalRiskIdentification from "../sequelize/TacticalRiskIdentificationSchema.js";
import { sendResponse } from "../helper/wrapper.js";

/**Function to create new Risk identification  */
export const tacticalRiskIdentificationCreateUpdate = async (req, res) => {
  const {
    id,
    risk_register_id,
    risk_owner,
    risk_name,
    risk_description,
    impact_description,
    causes_description,
    inherent_likelihood,
    inherent_consequence,
    inherent_risk_rating,
    inherent_risk_ranking,
    control_id,
    control_name,
    control_description,
    control_design_intent,
    control_effectiveness,
    control_owner,
    risk_rating,
    risk_ranking,
    opportunity_identification,
    opportunity_description,
    priority_confirmation,
    organization,
    department,
  } = req.body;

  req.body.causes_description = await encodeSingle_statement(
    causes_description
  );
  req.body.control_description = await encodeSingle_statement(
    control_description
  );
  req.body.impact_description = await encodeSingle_statement(
    impact_description
  );
  req.body.risk_description = await encodeSingle_statement(risk_description);

  req.body.opportunity_description =
    opportunity_description.length > 0
      ? JSON.stringify(
          await encodeTheEditorContent(opportunity_description, "description")
        )
      : "[{}]";

  /**Check record if organization is not coming then fetch organization according to department */
  let organizationId = organization;
  if (department) {
    const recordAccordingToOrganization =
      await getOrganizationAccordingToDepartment(department);
    organizationId = recordAccordingToOrganization[0].organization;
    req.body.organization = organizationId;
  }

  /**If id comes in body then it will update the query */
  if (id) {
    req.body.updated_by = req.user.sessionid;
    /**Update Strategic Risk Identification Query */
    const { query, values } = updateQueryBuilder(
      TacticalRiskIdentification,
      req.body
    );
    const updateStrategicRiskIdentification = await db.query(query, values);

    /**Insert Activity  */
    await insertActivityLog(req.user.sessionid, "update", "Roles", id);

    return sendResponse(res, 200, "Record Updated Successfully");
  } else {
    req.body.created_by = req.user.sessionid;
    /**Insert record for Strategic Risk Identification */
    const { query, values } = createQueryBuilder(
      TacticalRiskIdentification,
      req.body
    );
    const [createStrategicRiskIdentification] = await db.query(query, values);

    /**Insert record for activity log */
    await insertActivityLog(
      req.user.sessionid,
      "create",
      "Strategic Risk Identification",
      createStrategicRiskIdentification.insertId
    );

    return sendResponse(res, 200, "Record created successfully");
  }
};

/**Function to view all and single Tactical Risk Identification */
export const viewTacticalRiskIdentification = async (req, res) => {
  const { id } = req.params;
  const { approval_status, status = false } = req.query;

  const condition = await whereCondition({
    table: "tactical_risk_identification",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });

  const searchTableName = [
    "CONCAT(riskOwner.name , ' ' , riskOwner.surname)",
    "CONCAT(users.name , ' ' , users.surname)",
    "CONCAT(control.name , ' ' , control.surname)",
    "tactical_risk.unique_id",
    "tactical_risk.approval_status",
    "tactical_risk.risk_register_name",
    "tactical_risk.risk_context",
    "tactical_risk.status",
    "tactical_risk_identification.causes_description",
    "tactical_risk_identification.impact_description",
    "tactical_risk_identification.risk_description",
    "tactical_risk_identification.inherent_likelihood",
    "tactical_risk_identification.inherent_consequence",
    "tactical_risk_identification.inherent_risk_rating",
    "tactical_risk_identification.inherent_risk_ranking",
    "tactical_risk_identification.residual_risk_rating",
    "tactical_risk_identification.residual_risk_ranking",
    "tactical_risk_identification.opportunity_description",
    "tactical_risk_identification.risk_name",
    "organization.name",
  ];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = tactical_risk_identification.created_by",
    },
    {
      type: "left",
      targetTable: "users as control",
      onCondition: "control.id = tactical_risk_identification.control_owner",
    },
    {
      type: "left",
      targetTable: "users as riskOwner",
      onCondition: "riskOwner.id = tactical_risk_identification.risk_owner",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition:
        "organization.id = tactical_risk_identification.organization",
    },
    {
      type: "left",
      targetTable: "tactical_risk",
      onCondition:
        "tactical_risk.id = tactical_risk_identification.risk_register_id",
    },
  ];

  const joinsRecord = await makeJoins(joins);
  const statusQuery = status ? `AND tactical_risk.status = "${status}"` : "";
  const approvalFilter = approval_status
    ? `AND tactical_risk.approval_status = "${approval_status}"`
    : "";

  /**Record of all tacticalRiskIdentification */
  // const tacticalRiskIdentificationQuery = `SELECT tactical_risk_identification.id , tactical_risk_identification.risk_register_id , tactical_risk_identification.risk_owner , tactical_risk_identification.risk_name , tactical_risk_identification.risk_description , tactical_risk_identification.impact_description , tactical_risk_identification.causes_description , tactical_risk_identification.inherent_likelihood , tactical_risk_identification.inherent_consequence , tactical_risk_identification.inherent_risk_rating , tactical_risk_identification.inherent_risk_ranking , tactical_risk_identification.control_id , tactical_risk_identification.control_name , tactical_risk_identification.control_description , tactical_risk_identification.control_design_intent , tactical_risk_identification.control_effectiveness , tactical_risk_identification.control_owner , tactical_risk_identification.residual_risk_rating , tactical_risk_identification.residual_risk_ranking , tactical_risk_identification.opportunity_identification , tactical_risk_identification.opportunity_description , tactical_risk_identification.priority_confirmation , tactical_risk_identification.organization , tactical_risk_identification.department,tactical_risk_identification.created_by as created_by_id ,users.name as created_by , tactical_risk.risk_register_name, tactical_risk.approval_status , tactical_risk.status ,control.name as risk_control_owner_name,control.surname as risk_control_owner_surname, control.profile as risk_control_owner_profile , riskOwner.name as risk_owner_name,riskOwner.surname as risk_owner_surname , riskOwner.profile as risk_owner_profile , organization.name as organization_name ,users.surname as created_by_surname,users.profile as created_by_profile
  //   FROM tactical_risk_identification
  //   ${joinsRecord} where tactical_risk_identification.deleted = 0  ${searchCondition} ${condition}`;

  const tacticalRiskIdentificationQuery = `SELECT tactical_risk_identification.id , tactical_risk_identification.risk_register_id , tactical_risk_identification.risk_owner , tactical_risk_identification.risk_name , tactical_risk_identification.risk_description , tactical_risk_identification.impact_description , tactical_risk_identification.causes_description , tactical_risk_identification.inherent_likelihood , tactical_risk_identification.inherent_consequence , tactical_risk_identification.inherent_risk_rating , tactical_risk_identification.inherent_risk_ranking , tactical_risk_identification.control_id , tactical_risk_identification.tactical_identification , tactical_risk_identification.residual_risk_rating , tactical_risk_identification.residual_risk_ranking , tactical_risk_identification.opportunity_identification , tactical_risk_identification.opportunity_description , tactical_risk_identification.priority_confirmation , tactical_risk_identification.organization , tactical_risk_identification.department,tactical_risk_identification.created_by as created_by_id ,users.name as created_by , tactical_risk.risk_register_name,  tactical_risk.unique_id, tactical_risk.approval_status , tactical_risk.status ,control.name as risk_control_owner_name,control.surname as risk_control_owner_surname, control.profile as risk_control_owner_profile , riskOwner.name as risk_owner_name,riskOwner.surname as risk_owner_surname , riskOwner.profile as risk_owner_profile , organization.name as organization_name ,users.surname as created_by_surname,users.profile as created_by_profile
    FROM tactical_risk_identification 
    ${joinsRecord} where tactical_risk_identification.deleted = 0 ${statusQuery} ${approvalFilter}  ${searchCondition} ${condition}`;

  // Modify the ORDER BY clause to ASC
  let query = tacticalRiskIdentificationQuery.replace(
    /ORDER BY .*? DESC/,
    "ORDER BY tactical_risk_identification.residual_risk_ranking ASC"
  );

  let [tacticalRiskIdentification] = await db.query(query);
  // console.log(tacticalRiskIdentificationQuery);
  tacticalRiskIdentification = await decodeAndParseFields(
    tacticalRiskIdentification
  );
  // tacticalRiskIdentification.forEach(async (item) => {
  //   item.risk_description = await decodeSingle_statement(
  //     item.risk_description
  //   );
  //   item.causes_description = await decodeSingle_statement(
  //     item.causes_description
  //   );
  //   item.impact_description = await decodeSingle_statement(
  //     item.impact_description
  //   );

  //   item.control_description = await decodeSingle_statement(
  //     item.control_description
  //   );
  //   item.opportunity_description = await decodeTheEditorContent(
  //     item.opportunity_description,
  //     "description"
  //   );
  // });

  const totalRecord = await countQueryCondition(
    tacticalRiskIdentificationQuery
  );

  return sendResponse(res, 200, tacticalRiskIdentification, totalRecord);
};

export const updateTacticalRiskIdentificationPriority = async (req, res) => {
  const { id, priority_confirmation } = req.body;

  if (!id || !priority_confirmation) {
    return sendResponse(
      res,
      400,
      "Please provide id and priority_confirmation"
    );
  }

  let [riskIdentification] = await db.query(
    `SELECT risk_name, risk_register_id, risk_description, CONCAT(users.name , ' ' , users.surname) as risk_owner_name, users.email as risk_owner_email FROM tactical_risk_identification LEFT JOIN users ON users.id = tactical_risk_identification.risk_owner WHERE tactical_risk_identification.id = ? AND tactical_risk_identification.deleted = 0`,
    [id]
  );

  // const decodedDesc = await decodeSingle_statement(riskIdentification[0].risk_description)
  riskIdentification = await decodeAndParseFields(riskIdentification);

  const additionalData = {
    risk_name: riskIdentification[0].risk_name,
    risk_description: riskIdentification[0].risk_description,
    risk_owner_name: riskIdentification[0].risk_owner_name,
  };

  const { query, values } = updateQueryBuilder(
    TacticalRiskIdentification,
    req.body
  );
  const [updatePriority] = await db.query(query, values);

  if (updatePriority.affectedRows > 0) {
    const templateFileUrl = "mail_for_risk_prioritised_template.html";
    const templateName = "Risk Prioritised";
    const subject = `Priority Risk Notification - ${riskIdentification[0].risk_name}`;
    await riskRegisterTemplateHelper(
      templateFileUrl,
      templateName,
      subject,
      riskIdentification[0].risk_register_id,
      false,
      riskIdentification[0].risk_owner_email,
      additionalData
    );
    return sendResponse(res, 200, "Record updated successfully");
  } else {
    return sendResponse(res, 400, "Record not updated");
  }
};

/**Function to view all and single Tactical Risk Identification */
export const viewTacticalRiskIdentificationApprovalWorkflow = async (
  req,
  res
) => {
  const { id } = req.params;

  const condition = await whereCondition({
    table: "tactical_risk_identification",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });

  const searchTableName = [
    "CONCAT(riskOwner.name , ' ' , riskOwner.surname)",
    "CONCAT(users.name , ' ' , users.surname)",
    "CONCAT(approver.name , ' ' , approver.surname)",
    "tactical_risk.risk_register_name",
    "tactical_risk.risk_context",
    "tactical_risk.status",
    "meeting.meeting_title",
    "tactical_risk_identification.opportunity_description",
    "tactical_risk_identification.risk_name",
    "tactical_risk_identification.risk_description",
    "tactical_risk_identification.control_name",
    "tactical_risk_identification.control_description",
    "tactical_risk_identification.control_design_intent",
    "CONCAT(control.name , ' ' , control.surname)",
    "organization.name",
  ];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = tactical_risk_identification.created_by",
    },
    {
      type: "left",
      targetTable: "users as control",
      onCondition: "control.id = tactical_risk_identification.id",
    },
    {
      type: "left",
      targetTable: "users as riskOwner",
      onCondition: "riskOwner.id = tactical_risk_identification.risk_owner",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition:
        "organization.id = tactical_risk_identification.organization",
    },
    {
      type: "left",
      targetTable: "tactical_risk",
      onCondition:
        "tactical_risk.id = tactical_risk_identification.risk_register_id",
    },
  ];

  const joinsRecord = await makeJoins(joins);

  /**Record of all tacticalRiskIdentification */
  // const tacticalRiskIdentificationQuery = `SELECT tactical_risk_identification.id , tactical_risk_identification.risk_register_id , tactical_risk_identification.risk_owner , tactical_risk_identification.risk_name , tactical_risk_identification.risk_description , tactical_risk_identification.impact_description , tactical_risk_identification.causes_description , tactical_risk_identification.inherent_likelihood , tactical_risk_identification.inherent_consequence , tactical_risk_identification.inherent_risk_rating , tactical_risk_identification.inherent_risk_ranking , tactical_risk_identification.control_id , tactical_risk_identification.control_name , tactical_risk_identification.control_description , tactical_risk_identification.control_design_intent , tactical_risk_identification.control_effectiveness , tactical_risk_identification.control_owner , tactical_risk_identification.residual_risk_rating , tactical_risk_identification.residual_risk_ranking , tactical_risk_identification.opportunity_identification , tactical_risk_identification.opportunity_description , tactical_risk_identification.priority_confirmation , tactical_risk_identification.organization , tactical_risk_identification.department,tactical_risk_identification.created_by as created_by_id ,users.name as created_by , tactical_risk.risk_register_name, tactical_risk.approval_status , tactical_risk.status ,control.name as risk_control_owner_name,control.surname as risk_control_owner_surname, control.profile as risk_control_owner_profile , riskOwner.name as risk_owner_name,riskOwner.surname as risk_owner_surname , riskOwner.profile as risk_owner_profile , organization.name as organization_name ,users.surname as created_by_surname,users.profile as created_by_profile
  //   FROM tactical_risk_identification
  //   ${joinsRecord} where tactical_risk_identification.deleted = 0 AND tactical_risk.status = 'Complete' AND (tactical_risk.approval_status != 'Approved'  AND  tactical_risk.approval_status != 'Rejected')   ${searchCondition} ${condition}`;

  const tacticalRiskIdentificationQuery = `SELECT tactical_risk_identification.id , tactical_risk_identification.risk_register_id , tactical_risk_identification.risk_owner , tactical_risk_identification.risk_name , tactical_risk_identification.risk_description , tactical_risk_identification.impact_description , tactical_risk_identification.causes_description , tactical_risk_identification.inherent_likelihood , tactical_risk_identification.inherent_consequence , tactical_risk_identification.inherent_risk_rating , tactical_risk_identification.inherent_risk_ranking , tactical_risk_identification.control_id , tactical_risk_identification.tactical_identification , tactical_risk_identification.residual_risk_rating , tactical_risk_identification.residual_risk_ranking , tactical_risk_identification.opportunity_identification , tactical_risk_identification.opportunity_description , tactical_risk_identification.priority_confirmation , tactical_risk_identification.organization , tactical_risk_identification.department,tactical_risk_identification.created_by as created_by_id ,users.name as created_by , tactical_risk.risk_register_name,  tactical_risk.unique_id, tactical_risk.approval_status , tactical_risk.status ,control.name as risk_control_owner_name,control.surname as risk_control_owner_surname, control.profile as risk_control_owner_profile , riskOwner.name as risk_owner_name,riskOwner.surname as risk_owner_surname , riskOwner.profile as risk_owner_profile , organization.name as organization_name ,users.surname as created_by_surname,users.profile as created_by_profile
    FROM tactical_risk_identification 
    ${joinsRecord} where tactical_risk_identification.deleted = 0 AND tactical_risk.status = 'Complete' AND (tactical_risk.approval_status != 'Approved'  AND  tactical_risk.approval_status != 'Rejected')   ${searchCondition} ${condition}`;
  let [tacticalRiskIdentification] = await db.query(
    tacticalRiskIdentificationQuery
  );
  // console.log(tacticalRiskIdentificationQuery);
  tacticalRiskIdentification = await decodeAndParseFields(
    tacticalRiskIdentification
  );
  // tacticalRiskIdentification.forEach(async (item) => {
  //   item.risk_description = await decodeSingle_statement(
  //     item.risk_description
  //   );
  //   item.causes_description = await decodeSingle_statement(
  //     item.causes_description
  //   );
  //   item.impact_description = await decodeSingle_statement(
  //     item.impact_description
  //   );

  //   item.control_description = await decodeSingle_statement(
  //     item.control_description
  //   );
  //   item.opportunity_description = await decodeTheEditorContent(
  //     item.opportunity_description,
  //     "description"
  //   );
  // });

  const totalRecord = await countQueryCondition(
    tacticalRiskIdentificationQuery
  );

  return sendResponse(res, 200, tacticalRiskIdentification, totalRecord);
};

/**Function to delete a specific role */
export const deleteTacticalRiskIdentification = async (req, res) => {
  const { id } = req.params;
  const deleteRecordQuery = await deleteRecord(
    "tactical_risk_identification",
    id
  );
  // also deleted identification
  await deleteRecord("tactical_risk_identification", id, "risk_register_id");
  if (deleteRecordQuery) {
    /**Insert record for activity log */
    await insertActivityLog(
      req.user.sessionid,
      "delete",
      "Tactical Risk Identification",
      id
    );
    return sendResponse(res, 200, "Record deleted successfully");
  }
};

/**Function to delete a specific role */
export const approveRejectTacticalRiskIdentification = async (req, res) => {
  const { tacticalRiskIdentificationId, status, comment } = req.body;
  let messageStatus = status === "Approved" ? "approved" : "reject";
  const updateStrategicRiskIdentificationQuery = `UPDATE tactical_risk SET approval_status=?, reject_comment=?, updated_by=? WHERE id= ${tacticalRiskIdentificationId};`;
  const updateStrategicRiskIdentificationValues = [
    status,
    comment,
    req.user.sessionid,
  ];
  const [updateStrategicRiskIdentification] = await db.query(
    updateStrategicRiskIdentificationQuery,
    updateStrategicRiskIdentificationValues
  );

  /**Insert record for activity log */
  await insertActivityLog(
    req.user.sessionid,
    messageStatus,
    "Tactical Risk Identification",
    tacticalRiskIdentificationId
  );
  return sendResponse(res, 200, `Record ${messageStatus} successfully`);
};
