import db from "../db-config.js";
import StakeholderClassification from "../sequelize/StakeholderClassificationSchema.js";
import { sendResponse } from "../helper/wrapper.js";
import {
  countQueryCondition,
  createQueryBuilder,
  decodeAndParseFields,
  deleteRecord,
  encodeAndStringifyFields,
  getOrganizationAccordingToDepartment,
  insertActivityLog,
  makeJoins,
  searchConditionRecord,
  uniqueIdGenerator,
  updateQueryBuilder,
  whereCondition,
} from "../helper/general.js";

const relationshipType = {
  0: 1,
  1: 2,
};

const interactionScore = {
  Daily: 5,
  Weekly: 4,
  Monthly: 3,
  Quarterly: 2,
  Annually: 1,
};

const influenceScore = {
  High: 3,
  Medium: 2,
  Low: 1,
};

const connectionScore = {
  Strong: 3,
  Moderate: 2,
  Weak: 1,
};

// "10 - 20": 1,
// "21 - 25": 1,
// "26 - 29": 1,
// "30 - 35": 1,
// "36 - 45": 1,
// "46 - 60": 1,
const aggregateValues = [
  {
    start: -Infinity,
    end: 20,
    priority_level: "Low",
    values: "Keep in mind",
    actions: "Provide access to general channels of information and feedback",
  },
  {
    start: 21,
    end: 25,
    priority_level: "Medium",
    values: "Keep informed",
    actions: "Regularly update and consult to prevent escalation of concerns",
  },
  {
    start: 26,
    end: 29,
    priority_level: "High",
    values: "Keep informed",
    actions: "Monitor closely to prevent the development of threats",
  },
  {
    start: 30,
    end: 35,
    priority_level: "Medium",
    values: "Keep satisfied",
    actions: "Honor commitments, treat fairly, maintain regular communication",
  },
  {
    start: 36,
    end: 45,
    priority_level: "High",
    values: "Keep satisfied",
    actions: "Ensure high levels of engagement to avoid potential issues",
  },
  {
    start: 46,
    end: Infinity,
    priority_level: "High",
    values: "Keep close",
    actions:
      "Engage proactively to leverage opportunities and mitigate threats",
  },
];

export const createUpdateStakeholderClassification = async (req, res) => {
  const { id, department, stakeholder_category } = req.body;
  let total = 0;
  total += Number(relationshipType[req.body.relationship_type]);

  total += Number(interactionScore[req.body.interaction_frequency]);

  total += Number(influenceScore[req.body.influence_level]);

  total += Number(connectionScore[req.body.connection_strength]);

  let subTotal =
    (Number(req.body.interest_score) + Number(req.body.influence_score)) / 2;

  // TODO: Add some other unique check if needed
  const [existingClassification] = await db.query(
    `SELECT * FROM stakeholder_classification WHERE stakeholder_category = '${stakeholder_category}' AND deleted = 0`
  );

  if (
    existingClassification &&
    existingClassification.length &&
    id != existingClassification[0].id
  ) {
    return res.status(409).json({
      status: false,
      message: "Classification for this category already exists",
    });
  }


  let status = id ? "Update" : "Create";
  req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;

  if (department) {
    const organizationForTheDepartment =
      await getOrganizationAccordingToDepartment(department);

    if (!organizationForTheDepartment || !organizationForTheDepartment.length) {
      return res.status(400).json({
        status: false,
        message: "There is no organization for this department",
      });
    }

    req.body.organization = organizationForTheDepartment[0].organization;
  }

  if(!id){
    const unique_id = await uniqueIdGenerator(
      req.body.organization,
      department,
      "StakeHolderClassification",
      "stakeholder_classification",
      "unique_id",
      "unique_id"
    );
    req.body.unique_id = unique_id;
  }

  req.body = await encodeAndStringifyFields(req.body);
  const { query, values } = id
    ? updateQueryBuilder(StakeholderClassification, req.body)
    : createQueryBuilder(StakeholderClassification, req.body);
  // console.log(query, values);
  await db.query(query, values);
  await insertActivityLog(
    req.user.sessionid,
    status,
    "Stakeholder Classification",
    id
  );
  return sendResponse(res, 200, `Record ${status}d Successfully`);
};

export const getStakeholderClassification = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "stakeholder_classification",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    user: req.user,
  });
  const searchTableName = [
    "stakeholder_classification.classification",
    "stakeholder_classification.engagement_level",
    "stakeholder_classification.status",
    "stakeholder_classification.stakeholder_type",
    "stakeholder_classification.stakeholder_category",
    "stakeholder_classification.interaction_frequency",
    "stakeholder_classification.priority_level",
    "stakeholder_classification.connection_strength",
    "organization.name",
    "department.name",
    "CONCAT(users.name , ' ' , users.surname)",
  ];
  const searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = stakeholder_classification.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = stakeholder_classification.organization",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = stakeholder_classification.department",
    },
    // {
    //   type: "left",
    //   targetTable: "department as stake_department",
    //   onCondition:
    //     "stake_department.id = stakeholder_classification.stakeholder_department_internal",
    // },
    // {
    //   type: "left",
    //   targetTable: "users as u1",
    //   onCondition:
    //     "u1.id = stakeholder_classification.stakeholder_name_internal",
    // },
  ];
  const joinRecord = await makeJoins(joins);

  const stakeholderClassificationQuery = `SELECT stakeholder_classification.* , organization.name as organization_name , department.name AS department_name , users.name AS created_by_name , users.surname AS created_by_surname , users.profile AS created_by_profile FROM stakeholder_classification ${joinRecord} WHERE stakeholder_classification.deleted = 0 ${searchCondition} ${condition}`;

  let [stakeholderClassification] = await db.query(
    stakeholderClassificationQuery
  );
  stakeholderClassification = await decodeAndParseFields(
    stakeholderClassification
  );

  // for (const stakeholder of stakeholderClassification) {
  //   if (stakeholder.stakeholder_type == 0) {
  //     const employee = stakeholder.stakeholder_name_internal;

  //     const [employeeName] = await db.query(
  //       `SELECT CONCAT(name , ' ' , surname) AS name , role FROM users WHERE id = ${employee}`
  //     );
  //     const [roles] = await db.query(
  //       `SELECT name FROM roles WHERE id = ${employeeName[0].role}`
  //     );
  //     stakeholder.stakeholder_name = employeeName[0].name;
  //     stakeholder.role_name = roles[0].name;
  //   }
  // }
  const totalRecord = await countQueryCondition(stakeholderClassificationQuery);

  return sendResponse(res, 200, stakeholderClassification, totalRecord);
};

export const deleteStakeholderClassification = async (req, res) => {
  const { id } = req.params;
  await deleteRecord(StakeholderClassification, id);
  await insertActivityLog(
    req.user.sessionid,
    "delete",
    "Stakeholder Classification",
    id
  );
  return sendResponse(res, 200, "Record deleted successfully");
};
