import asyncHandler from "express-async-handler";
import db from "../db-config.js";
import {
  insertActivityLog,
  makeJoins,
  encodeSingle_statement,
  decodeSingle_statement,
  deleteSettingRecord,
  countQueryCondition,
  settingsUpdater,
  createQueryBuilder,
  whereCondition,
  searchConditionRecord,
  uniqueIdGenerator,
  decodeAndParseFields,
  getFilterConditions,
  getListingData,
  getRecord,
  getFirstCreatedAndAllOrganizationIds,
} from "../helper/general.js";
import { sendResponse } from "../helper/wrapper.js";
import Skill from "../sequelize/SkillSchema.js";
import { updateQueryBuilder } from "../helper/queryBuilder.js";

/**function to create and update Skill */
export const createUpdateSkill = async (req, res) => {
  const { name, description, organization, id } = req.body;
  if (!id) {
    const result = await getFirstCreatedAndAllOrganizationIds();

    if (!result.firstOrgId) {
      return sendResponse(res, 404, "No organization found.");
    }
    const unique_id = await uniqueIdGenerator(
      result.firstOrgId,
      req.body?.department || null,
      "Skills",
      "skills",
      "unique_id",
      "unique_id"
    );
    req.body.unique_id = unique_id;
    req.body.organization =
      organization.length > 0 ? organization : result.organizationIds;
  }
  req.body.skills_type = req.body.skill_type;
  const encodedDescription = await encodeSingle_statement(description);
  req.body.description = encodedDescription;
  req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;
  /**If id comes in body then it will update the query */
  const [checkNameWithOrganization] = await getRecord("skills", "name", name);
  if (checkNameWithOrganization) {
    if (id && checkNameWithOrganization.id != id) {
      return sendResponse(res, 400, "Skill name already exist");
    } else if (!id) {
      return sendResponse(res, 400, "Skill name already exist");
    }
  }
  if (id) {
    /**Check that this organization have that particular name or not*/
    if (organization.length === 0) {
      return sendResponse(
        res,
        400,
        "Please Select at least one business Structure"
      );
    }

    /**update query */
    const { query, values } = updateQueryBuilder(Skill, req.body);
    await db.query(query, values);
    /**Update record for activity log */
    await insertActivityLog(req.user.sessionid, "update", "Skill", id);
    return sendResponse(res, 200, "Record updated successfully");
  } else {
    const { query, values } = createQueryBuilder(Skill, req.body);
    const result = await db.query(query, values);
    /**Insert record for activity log */
    await insertActivityLog(
      req.user.sessionid,
      "create",
      "Skill",
      result.insertId
    );
    return sendResponse(res, 200, "Record created successfully");
  }
};

/**function to get all Skill with name or specific Skill */
export const getSkill = async (req, res) => {
  let { organizationCondition, condition } = await getFilterConditions(
    req,
    db,
    "skills"
  );
  /** If value come with any search condition then search that word */
  const searchTableName = [
    "skills.name",
    "skills.description",
    "createdUser.name",
  ];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );
  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users as createdUser",
      onCondition: "createdUser.id = skills.created_by",
    },
  ];
  const joinsRecord = await makeJoins(joins);
  /**Record of all Skill */
  let skillQuery = `SELECT 
  skills.*, 
  skills.organization AS organizations, 
  createdUser.name AS created_by, 
  createdUser.surname AS created_by_surname, 
  createdUser.id AS created_by_id, 
  createdUser.profile AS created_by_profile
FROM skills
${joinsRecord} 
WHERE skills.deleted = 0 ${searchCondition} ${organizationCondition} ${condition}`;
  let [skills] = await db.query(skillQuery);
  skills = await getListingData(skills);
  /**Count all organization */
  const totalRecord = await countQueryCondition(skillQuery);
  return sendResponse(res, 200, skills, totalRecord);
};

/**Function to delete a specific Skill */
export const deleteSkill = async (req, res) => {
  const { id } = req.params;
  const deleteRecordQuery = await deleteSettingRecord("skills", id);
  if (deleteRecordQuery) {
    /**Insert record for activity log */
    await insertActivityLog(req.user.sessionid, "delete", "skills", id);
    return sendResponse(res, 200, "Record deleted successfully");
  }
};

/**function to get Skills according to organization */
// export const getSkillsAccordingToOrg = asyncHandler(async (req, res) => {
//     try {
//         const { organization } = req.params;
//         const Query = `SELECT id,name,description,skills_type,industry,organization FROM skills
//         where deleted = 0 AND organization = ${organization} ORDER BY id DESC`;
//         const Result = await db.query(Query);

//         return res.status(200).json({ status: true, data: Result });
//     } catch (error) {
//         // Check if an error occurs, then send that error to the log file
//         storeError(error);
//         return res.status(500).json({ status: false, message: error.message });
//     }
// });
