import db from "../db-config.js";
import {
  countQueryCondition,
  createQueryBuilder,
  decodeAndParseFields,
  deleteRecord,
  encodeAndStringifyFields,
  getOrganizationAccordingToDepartment,
  insertActivityLog,
  makeJoins,
  processUploadDocuments,
  searchConditionRecord,
  uniqueIdGenerator,
  updateQueryBuilder,
  uploadFile,
  whereCondition,
} from "../helper/general.js";
import { sendResponse } from "../helper/wrapper.js";
import ProjectRegistrationDocs from "../sequelize/ProjectRegistrationDocsSchema.js";
import ProjectRegistration from "../sequelize/ProjectRegistrationSchema.js";

export const createUpdateProjectRegistration = async (req, res) => {
  req.body = (await decodeAndParseFields([req.body]))[0];

  const { id, department, sidebar_id = 322 } = req.body;

  let status = id ? "Updated" : "Created";

  req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;

  if (department) {
    req.body.organization = (await getOrganizationAccordingToDepartment(department))[0].organization;
  }

  const documents = await processUploadDocuments(req, sidebar_id);

  if (id && documents.length) {
    const deleteQuery = `UPDATE project_registration_docs SET deleted = 1 where deleted = 0 AND project_registration_id = ?`;
    await db.query(deleteQuery, [id]);
  }

  req.body = await encodeAndStringifyFields(req.body);

  // return console.log("req.body: ", req.body);
  if (!id) {
    const unique_id = await uniqueIdGenerator(
      req.body.organization,
      department,
      "ProjectRegistration",
      "project_registration",
      "unique_id",
      "unique_id"
    );
    req.body.unique_id = unique_id;
  }

  const { query, values } = id
    ? updateQueryBuilder(ProjectRegistration, req.body)
    : createQueryBuilder(ProjectRegistration, req.body);

  const [result] = await db.query(query, values);

  if (documents || Array.isArray(documents)) {
    for (const file of documents) {
      const title = file.title; // Use file name as title
      const ddrm_id = file?.ddrm_id;
      const id_doc = file.id;
      // Move the file to the public folder
      // Save file details for database
      const { query: docQuery, values: docValues } = createQueryBuilder(ProjectRegistrationDocs, {
        doc_no: id_doc,
        project_registration_id: id ? id : result.insertId,
        title,
        ddrm_id,
      });

      await db.query(docQuery, docValues);
    }
  }

  await insertActivityLog(req.user.sessionid, status, "Project Registration", id ? id : result.insertId);

  return sendResponse(res, 200, `Record ${status} successfully`);
};

export const getProjectRegistration = async (req, res) => {
  const { id } = req.params;

  const condition = await whereCondition({
    table: "project_registration",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    user: req.user,
  });

  const searchTableName = [
    "project_registration.cipc_registration_number",
    "project_registration.site_address",
    "project_registration.project_title",
    "project_registration.contract_reference_number",
    "project_registration.project_value",
    "project_registration.contract_value",
    "project_registration.currency",
    "project_registration.classification_value",
    "project_registration.classification_criticality",
    "project_registration.classification_category",
    "project_registration.access_location",
    "project_registration.project_description",
    "project_registration.type_of_work",
    "project_registration.reason",
    "project_registration.classification_impact",
    "project_registration.project_members",
    "project_registration.sub_contractors_personnel",
    "organization.name",
    "department.name",
    "cr.contractor_name",
    "subContractor.contractor_name",
  ];

  const searchCondition = await searchConditionRecord(req.query.search, searchTableName);

  const joins = [
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = project_registration.organization",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = project_registration.department",
    },
    {
      type: "left",
      targetTable: "contractor_registration AS cr",
      onCondition: "cr.id = project_registration.contractor",
    },
    {
      type: "left",
      targetTable: "contractor_registration AS subContractor",
      onCondition: "subContractor.id = project_registration.sub_contractor",
    },
    {
      type: "left",
      targetTable: "type_of_service",
      onCondition: "type_of_service.id = project_registration.type_of_work",
    },
    {
      type: "left",
      targetTable: "currency",
      onCondition: "currency.id = project_registration.currency",
    },
  ];

  const joinCondition = await makeJoins(joins);

  const projectRegistrationQuery = `SELECT project_registration.* , type_of_service.name AS type_of_work_name , organization.name AS organization_name , department.name AS department_name , cr.contractor_name AS contractor_name  , subContractor.contractor_name AS subContractor_contractor_name, currency.name AS currency_name FROM project_registration ${joinCondition} WHERE project_registration.deleted = 0 ${searchCondition} ${condition}`;

  let [projectRegistration] = await db.query(projectRegistrationQuery);

  projectRegistration = await decodeAndParseFields(projectRegistration);

  for (const project of projectRegistration) {
    const [files] = await db.query(
      `SELECT project_registration_docs.*, repository.url as file, document_creation.document_name as title FROM project_registration_docs LEFT JOIN repository ON repository.id = project_registration_docs.ddrm_id LEFT JOIN document_creation ON document_creation.id = repository.document_creation_id WHERE project_registration_docs.deleted = 0 AND project_registration_id = ${project.id}`
    );

    project.upload_documents = files;
    let members = project.project_members;
    for (const member of members) {
      const [roles] = await db.query(`SELECT * from role_on_the_project WHERE id = ?`, [member.role_on_the_project]);
      member.role_on_the_project_name = roles[0]?.name;
    }
    let personnel = project.sub_contractors_personnel;
    for (const sub_contractor of personnel) {
      const [roles] = await db.query(`SELECT * from role_on_the_project WHERE id = ?`, [
        sub_contractor.role_on_the_project,
      ]);
      sub_contractor.role_on_the_project_name = roles[0]?.name;
    }
  }

  const totalRecord = await countQueryCondition(projectRegistrationQuery);

  return sendResponse(res, 200, projectRegistration, totalRecord);
};

export const deleteProjectRegistration = async (req, res) => {
  const { id } = req.params;
  await deleteRecord(ProjectRegistration, id);

  await insertActivityLog(req.user.sessionid, "delete", "Project Registration", id);

  return sendResponse(res, 200, "Record deleted successfully");
};
