import asyncHandler from "express-async-handler";
import db from "../db-config.js";
import {
  countQueryCondition,
  createQueryBuilder,
  deleteRecord,
  getCompanyName,
  getOrganizationAccordingToDepartment,
  getUserById,
  insertActivityLog,
  makeJoins,
  makeLoopAndGetMultipleUsersDetails,
  searchConditionRecord,
  storeError,
  uniqueIdGenerator,
  updateQueryBuilder,
  whereCondition,
} from "../helper/general.js";
import { createCalendar } from "../helper/calendar.js";
import { fileURLToPath } from "url";
import path from "path";
import fs from "fs";
import Meeting from "../sequelize/MeetingSchema.js";
import { sendResponse } from "../helper/wrapper.js";
import sendEmail from "../helper/sendEmail.js";
import moment from "moment";
import ical from "ical-generator";
// root directory path of the project
const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

/**Function to create and update Meeting   */
export const createUpdateMeeting = async (req, res) => {
  const { id, organization, department, meeting_owner, participants } = req.body;

  /**Check record if organization is not coming then fetch organization according to department */
  let organizationId = organization;
  if (department) {
    const recordAccordingToOrganization =
      await getOrganizationAccordingToDepartment(department);
    organizationId = recordAccordingToOrganization[0].organization;
    req.body.organization = organizationId;
  }
  /**If id comes in body then it will update the query */
  if (id) {
    /**Update Roles Query */
    req.body.created_by = req.user.sessionid;
    const { query, values } = updateQueryBuilder(Meeting, req.body);
    await db.query(query, values);
    /**Insert Activity  */
    insertActivityLog(req.user.sessionid, "update", "Meeting", id);

    return sendResponse(res, 200, "Record updated successfully");
  } else {
    const unique_id = await uniqueIdGenerator(
      organizationId,
      department,
      "MS",
      "meeting",
      "unique_id",
      "unique_id"
    );
    req.body.unique_id = unique_id;
    const meetingId = Math.floor(1000 + Math.random() * 9000);
    req.body.created_by = req.user.sessionid;
    req.body.meeting_id = meetingId;
    /**Insert record for Meeting */

    const { query, values } = createQueryBuilder(Meeting, req.body);
    const [createMeeting] = await db.query(query, values);

    const meetingData = await getMeetingById(createMeeting.insertId)

    let agendaData = [];
    meetingData[0].agenda.forEach((agenda) => {
      agendaData.push({
        title: agenda.title,
        allocated_name: agenda.allocated_name,
      });
    });
    const companyDetails = await getCompanyName(meetingData[0].organization)

    const startDate = moment(meetingData[0].planned_meeting_date_from).format("YYYY-MM-DD");
    const startTime = moment(meetingData[0].planned_meeting_date_from).format("hh:mm A");
    const endDate = moment(meetingData[0].planned_meeting_date_to).format("YYYY-MM-DD");
    const endTime = moment(meetingData[0].planned_meeting_date_to).format("hh:mm A");

    // send email to participants and meeting owner 
    const meetingOwner = await getUserById(meeting_owner)

    let participantDetails = [];
    for (let i = 0; i < participants.length; i++) {
      const data = await getUserById(participants[i])
      if (data !== null && data !== undefined) {
        participantDetails.push({
          name: `${data?.name}`.trim(),
          email: `${data?.email}`.trim(),
        });
      }
    }

    let chairPerson = `${meetingData[0].host_name} ${meetingData[0].host_surname}`;
    const fileData = await createMeetingICSFile(meetingData, meetingOwner, participantDetails)

    const sendMeetingOwnerRecordArray = {
      templateFileUrl: "mail_for_meeting_scheduled_template.html",
      templateName: "Meeting Scheduled",
      meeting_title: meetingData[0]?.meeting_title,
      name: meetingOwner.name,
      meeting_creator: meetingOwner.name,
      meeting_chairperson: chairPerson,
      start_date: startDate,
      start_time: startTime,
      end_date: endDate,
      end_time: endTime,
      meeting_venue: meetingData[0]?.venue,
      meeting_hierarchy: meetingData[0]?.meeting_hierarchy,
      meeting_occurrence: startDate,
      venue_type_name: "Offline",
      meeting_link_or_physical_venue_name: meetingData[0]?.venue,
      participantDetails: participantDetails
        .map(participant => `${participant?.name}`)
        .join("<br>"),
      agendaData: agendaData
        .map(agenda => `${agenda.title} - ${agenda.allocated_name}`)
        .join("<br>"),
      meeting_creator_email: meetingOwner.email,
      company_name: companyDetails?.name,
      company_website: companyDetails?.business_logo,
    };

    const meetingOwnerInfo = await sendEmail(
      "info@harmonyandhelp.com",
      meetingOwner.email,
      `Meeting Scheduled: ${meetingData[0]?.meeting_title}`,
      sendMeetingOwnerRecordArray,
      fileData
    );
    for (let user of participantDetails) {
      const sendRecordArray = {
        templateFileUrl: "mail_for_meeting_scheduled_template.html",
        templateName: "Meeting Scheduled",
        meeting_title: meetingData[0]?.meeting_title,
        name: `${user?.name}`.trim(),
        meeting_creator: meetingOwner.name,
        meeting_chairperson: chairPerson,
        start_date: startDate,
        start_time: startTime,
        end_date: endDate,
        end_time: endTime,
        meeting_venue: meetingData[0]?.venue,
        meeting_hierarchy: meetingData[0]?.meeting_hierarchy,
        meeting_occurrence: startDate,
        venue_type_name: "Offline",
        meeting_link_or_physical_venue_name: meetingData[0]?.venue,
        participantDetails: participantDetails
          .map(participant => `${participant?.name}`)
          .join("<br>"),
        agendaData: agendaData
          .map(agenda => `${agenda.title} - ${agenda.allocated_name}`)
          .join("<br>"),
        meeting_creator_email: meetingOwner.email,
        company_name: companyDetails?.name,
        company_website: companyDetails?.business_logo,
      };

      const info = await sendEmail(
        "info@harmonyandhelp.com",
        user.email,
        `Meeting Scheduled: ${meetingData[0]?.meeting_title}`,
        sendRecordArray,
        fileData,
      );
    }

    /**Insert record for activity log */
    insertActivityLog(
      req.user.sessionid,
      "create",
      "Meeting",
      createMeeting.insertId
    );
    return sendResponse(res, 200, "Record created successfully");
  }
};
/**Function to view all  meeting */
export const viewAllMeeting = asyncHandler(async (req, res) => {
  const condition = await whereCondition({
    table: "meeting",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id: req.params.id,
    user: req.user,
  });
  /** If value come with any search condition then search that word */
  const searchTableName = [
    "meeting.meeting_title",
    "meeting.venue",
    "meeting.meeting_hierarchy",
    "organization.name",
    "meeting.meeting_title",
    "CONCAT(u1.name , ' ' , u1.surname)",
  ];
  const searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );
  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users as createdUser",
      onCondition: "createdUser.id = meeting.created_by",
    },
    {
      type: "left",
      targetTable: "users as u1",
      onCondition: "u1.id = meeting.meeting_owner",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = meeting.organization",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = meeting.department",
    },
  ];
  const joinsRecord = await makeJoins(joins);
  /**Record of all meetings */
  const meetingsQuery = `SELECT meeting.*, CONCAT(u1.name , ' ' , u1.surname) as meeting_owner_name, meeting.created_by as created_by_id,createdUser.name as created_by , createdUser.surname as created_by_surname, createdUser.profile as created_by_profile , u1.name as host_name, u1.surname as host_surname,u1.profile as host_profile , organization.name as organization_name FROM meeting
     ${joinsRecord} WHERE meeting.deleted = 0 ${searchCondition} ${condition}`;
  const [meetings] = await db.query(meetingsQuery);
  for (const meeting of meetings) {
    const agendas = JSON.parse(meeting?.agenda || `[]`);
    for (const agenda of agendas) {
      const userId = agenda.allocated_to;
      let userName = "";
      if (userId) {
        [userName] = await db.query(
          `SELECT CONCAT(name , ' ' , surname) AS name, profile FROM users WHERE id = ${userId}`
        );
      }
      agenda.allocated_name = userName[0]?.name;
      agenda.allocated_profile = userName[0]?.profile;
    }
    meeting.agenda = agendas;
  }
  const totalRecord = await countQueryCondition(meetingsQuery);
  /**Make loop for getting record of teams members details according to team leaders */
  await makeLoopAndGetMultipleUsersDetails(meetings, "attendee");
  return sendResponse(res, 200, meetings, totalRecord);
});
/**Function to delete a specific Meeting */
export const deleteMeeting = asyncHandler(async (req, res) => {
  try {
    const { id } = req.params;
    const deleteRecordQuery = await deleteRecord("meeting", id);
    if (deleteRecordQuery) {
      /**Insert record for activity log */
      insertActivityLog(req.user.sessionid, "delete", "Meeting", id);
      return res.status(200).json({
        status: true,
        message: "Record deleted successfully",
      });
    }
  } catch (error) {
    /** Check if error is come then send that error to log file  */
    storeError(error);
    res.status(500).json({
      status: false,
      message: error.message,
    });
  }
});
export const viewAllMeetingCalendar = asyncHandler(async (req, res) => {
  try {
    const { id } = req.params;
    const condition = await whereCondition({
      table: "meeting",
      page: req.query.page,
      all: req.query.all,
      pageSize: req.query.pageSize,
      filter: req.query.filter,
      id,
      grouped: req.query.grouped,
      user: req.user,
    });

    /** If value come with any search condition then search that word */
    let searchCondition = req.query.search
      ? `AND (meeting.meeting_title LIKE '%${req.query.search}%')`
      : "";
    /**Make Joins according to tables */
    const joins = [
      {
        type: "left",
        targetTable: "users",
        onCondition: "users.id = meeting.created_by",
      },
      {
        type: "left",
        targetTable: "users as u1",
        onCondition: "u1.id = meeting.meeting_owner",
      },
      {
        type: "left",
        targetTable: "organization",
        onCondition: "organization.id = meeting.organization",
      },
      {
        type: "left",
        targetTable: "department",
        onCondition: "department.id = meeting.department",
      },
    ];
    const joinsRecord = await makeJoins(joins);
    /**Record of all meetings */
    const meetingsQuery = `SELECT organization.name as organization_name, meeting.id,meeting.meeting_id,meeting.meeting_title,meeting.meeting_hierarchy,meeting.participants,meeting.meeting_owner,meeting.planned_meeting_date_from,meeting.planned_meeting_date_to,meeting.venue,meeting.agenda,meeting.organization,meeting.department,meeting.created_by as created_by_id,u1.name as host_name,users.name as created_by
     FROM meeting
     ${joinsRecord} where meeting.deleted = 0 ${searchCondition} ${condition}`;
    const meetings = await db.query(meetingsQuery);
    // /**Check multiple users organization */
    // const organization = await inWhereFormat(req.additionalData);
    // /**Check if login person is super admin of that organization then show all record */
    // const isSuperAdmin =
    //   req.user.sessionidRole == 0
    //     ? `AND meeting.organization in ${organization}`
    //     : `AND meeting.organization in ${organization}`;
    // /**filter on the basis of leader or not */
    // let leaderFilter = await isTeamLeader(req.user.sessionid, "meeting", "meeting");
    // /**Record of all meeting */
    // let meetingQuery = `SELECT planned_meeting_date_from,planned_meeting_date_to,meeting_type.description,meeting_title,venue from
    // meeting left join meeting_type on meeting_type.id = meeting.meetingType
    // left join users on users.id = meeting.created_by
    // left join organization on organization.id = meeting.organization
    // where meeting.deleted = 0 ${isSuperAdmin} ${leaderFilter} order by meeting.id desc `;
    // const meeting = await db.query(meetingQuery);
    const fileObj = await createCalendar(meetings);
    const completePath = path.join(__dirname, "../", fileObj?.filePath);
    // Check if the file exists
    if (!fs.existsSync(completePath)) {
      return res.status(404).json({
        status: false,
        message: "File not found",
      });
    }
    return res.status(200).json({
      status: true,
      data: fileObj.fileName,
    });
  } catch (error) {
    /** Check if error is come then send that error to log file  */
    storeError(error);
    res.status(500).json({
      status: false,
      message: error.message,
    });
  }
});

export const getMeetingById = asyncHandler(async (meetingId) => {
  const condition = await whereCondition({
    table: "meeting",
    id: meetingId,
    // user: req.user,
  });
  /** If value come with any search condition then search that word */
  const searchTableName = [
    "meeting.meeting_title",
    "meeting.venue",
    "meeting.meeting_hierarchy",
    "organization.name",
    "meeting.meeting_title",
    "meeting.meeting_title",
    "CONCAT(u1.name , ' ' , u1.surname)",
  ];
  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users as createdUser",
      onCondition: "createdUser.id = meeting.created_by",
    },
    {
      type: "left",
      targetTable: "users as u1",
      onCondition: "u1.id = meeting.meeting_owner",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = meeting.organization",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = meeting.department",
    },
  ];
  const joinsRecord = await makeJoins(joins);
  const meetingsQuery = `SELECT meeting.*, meeting.created_by as created_by_id,
  createdUser.name as created_by , createdUser.surname as created_by_surname, 
  createdUser.profile as created_by_profile , u1.name as host_name, 
  u1.surname as host_surname,u1.profile as host_profile , 
  organization.name as organization_name FROM meeting
     ${joinsRecord} WHERE meeting.deleted = 0 ${condition}`;
  const [meetings] = await db.query(meetingsQuery);
  for (const meeting of meetings) {
    const agendas = JSON.parse(meeting?.agenda || `[]`);
    for (const agenda of agendas) {
      const userId = agenda.allocated_to;
      let userName = "";
      if (userId) {
        [userName] = await db.query(
          `SELECT CONCAT(name , ' ' , surname) AS name, profile FROM users WHERE id = ${userId}`
        );
      }
      agenda.allocated_name = userName[0]?.name;
      agenda.allocated_profile = userName[0]?.profile;
    }
    meeting.agenda = agendas;
  }
  await makeLoopAndGetMultipleUsersDetails(meetings, "attendee");
  return meetings;
})

export async function createMeetingICSFile(meetingData, meetingOwner, participantDetails) {
  const {
    meeting_title,
    planned_meeting_date_from,
    planned_meeting_date_to,
    venue,
    agenda
  } = meetingData[0];

  // Create calendar instance
  const calendar = ical({
    name: 'Meeting Calendar',
    prodId: { company: "Harmony and Help", product: "SSHEQ" },
  });

  const startDateTime = new Date(planned_meeting_date_from);
  const endDateTime = new Date(planned_meeting_date_to);

  // Create attendees array from participant details
  const attendees = participantDetails.map(participant => ({
    name: participant?.name,
    email: participant?.email,
    rsvp: true
  }));

  // Add meeting owner/organizer
  attendees.push({
    name: meetingOwner?.name,
    email: meetingOwner?.email,
    rsvp: true
  });

  // Create agenda description
  let description = 'Agenda:\n';
  if (Array.isArray(agenda) && agenda.length > 0) {
    agenda.forEach((agenda, index) => {
      description += `${index + 1}. ${agenda.title}\n`;
      if (agenda.description) {
        description += `   ${agenda.title}\n`;
      }
    });
  } else {
    description += 'No agenda items specified.';
  }

  // Add event to calendar
  calendar.createEvent({
    start: startDateTime,
    end: endDateTime,
    summary: meeting_title,
    description: description,
    location: venue,
    organizer: {
      name: meetingOwner.name,
      email: meetingOwner.email
    },
    attendees: attendees?.map(attendee => ({
      name: attendee.name,
      email: attendee.email,
      rsvp: true,   // Explicitly add RSVP
      role: 'REQ-PARTICIPANT' // Specify role
    })),
  });

  const calendarDir = path.join(process.cwd(), 'public', 'calendar');
  if (!fs.existsSync(calendarDir)) {
    fs.mkdirSync(calendarDir, { recursive: true });
  }

  const fileName = `meeting_${moment().format('YYYYMMDD_HHmmss')}.ics`;
  const filePath = path.join(calendarDir, fileName);

  fs.writeFileSync(filePath, calendar.toString());

  const fileToUpload = {
    uri: `${filePath}`,
    name: fileName,
    type: 'text/calendar'
  };

  return fileToUpload;
}
