import db from "../db-config.js";
import {
  countQueryCondition,
  decodeSingle_statement,
  deleteSettingRecord,
  encodeSingle_statement,
  insertActivityLog,
  makeJoins,
  whereCondition,
  settingsUpdater,
  searchConditionRecord,
  createQueryBuilder,
  uniqueIdGenerator,
  getFirstCreatedAndAllOrganizationIds,
  encodeAndStringifyFields,
  updateQueryBuilder,
  getFilterConditions,
  getListingData,
} from "../helper/general.js";
import IllnessType from "../sequelize/IllnessTypeSchema.js";
import { sendResponse } from "../helper/wrapper.js";

/**Function to create new Incident Category for employee  */
export const createUpdateIllnessType = async (req, res) => {
  const { id, organization } = req.body;

  if (id && organization.length === 0) {
    return sendResponse(res, 404, "Please Select at least one business Structure");
  }

  if (!id) {
    const result = await getFirstCreatedAndAllOrganizationIds();

    if (!result.firstOrgId) {
      return sendResponse(res, 404, "No organization found.");
    }
    const unique_id = await uniqueIdGenerator(
      result.firstOrgId,
      req.body?.department || null,
      "ODIT",
      "illness_type",
      "unique_id",
      "unique_id"
    );
    req.body.unique_id = unique_id;
    req.body.organization = organization.length > 0 ? organization : result.organizationIds;
  }

  let status = id ? "Updated" : "Created";
  req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;
  req.body = await encodeAndStringifyFields(req.body);
  const { query, values } = id ? updateQueryBuilder(IllnessType, req.body) : createQueryBuilder(IllnessType, req.body);
  const [result] = await db.query(query, values);
  await insertActivityLog(req.user.sessionid, status, "IllnessType", id ? id : result.insertId);
  return sendResponse(res, 200, `Record ${status} Successfully`);
};

/**Function to view all and single Incident Category */
export const viewIllnessType = async (req, res) => {
  let { organizationCondition, condition } = await getFilterConditions(req, db, "illness_type");

  // const { id } = req.params;
  // const { dropdown } = req.query;
  // const condition = await whereCondition({
  //   table: "illness_type",
  //   page: req.query.page,
  //   all: req.query.all,
  //   orderBy: req.query.orderBy,
  //   pageSize: req.query.pageSize,
  //   filter: req.query.filter,
  //   id,
  //   grouped: req.query.grouped,
  //   user: req.user,
  // });

  const searchTableName = [
    "illness_type.name",
    "illness_type.description",
    "CONCAT(users.name , ' ' , users.surname)",
  ];
  let searchCondition = await searchConditionRecord(req.query.search, searchTableName);

  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = illness_type.created_by",
    },
    // {
    //   type: "left",
    //   targetTable: "organization",
    //   onCondition: "organization.id = illness_type.organization",
    // },
    {
      type: "left",
      targetTable: "illness_type AS parent_illness_type",
      onCondition: "parent_illness_type.id = illness_type.parent",
    },
  ];
  const joinsRecord = await makeJoins(joins);

  let IllnessTypeQuery = `SELECT illness_type.id, illness_type.unique_id, illness_type.created_by as created_by_id, illness_type.name,parent_illness_type.name AS parent_name , illness_type.description, users.name as created_by, users.surname as created_by_surname, users.profile as created_by_profile, illness_type.organization as organizations , illness_type.organization, illness_type.parent FROM illness_type ${joinsRecord} WHERE illness_type.deleted = 0 ${searchCondition} ${organizationCondition} ${condition}`;
  // IllnessTypeQuery = IllnessTypeQuery.replace("ORDER BY illness_type.id DESC", "ORDER BY illness_type.id ASC");
  let [IllnessType] = await db.query(IllnessTypeQuery);

  IllnessType = await getListingData(IllnessType);

  async function fetchSubChildren(parentId) {
    const [subChildren] = await db.query(`SELECT * FROM illness_type WHERE deleted = 0 AND parent = ?`, [parentId]);

    // Only populate sub-children if any are found, and no need to recurse if not needed
    for (const child of subChildren) {
      child.description = await decodeSingle_statement(child.description);
      child.sub_children = await fetchSubChildren(child.id);
    }
    return subChildren;
  }

  // Map through IllnessType and only populate the parent along with its sub-children.
  const nestedIllnessTypes = [];

  for (const item of IllnessType) {
    item.description = await decodeSingle_statement(item.description);
    if (item.parent === null) {
      item.sub_children = await fetchSubChildren(item.id);
      nestedIllnessTypes.push(item); // Push only the parent with its nested structure
    }
  }

  const totalRecord = await countQueryCondition(IllnessTypeQuery);
  return sendResponse(res, 200, nestedIllnessTypes, totalRecord);
};

/**Function to delete a specific Incident Category */
export const deleteIllnessType = async (req, res) => {
  const { id } = req.params;
  const deleteRecordQuery = await deleteSettingRecord("illness_type", id);
  if (deleteRecordQuery) {
    /**Insert record for activity log */
    await insertActivityLog(req.user.sessionid, "delete", "illness_type", id);
    return sendResponse(res, 200, "Record deleted successfully");
  }
};

export const getParentIllnessTypeList = async (req, res) => {
  let { filter } = req.query;
  filter = typeof filter === "string" ? JSON.parse(filter) : filter;
  let { organizations } = filter;
  console.log(organizations);
  if (!organizations || organizations.length == 0) {
    const [arrOrg] = await db.query(`SELECT id FROM organization WHERE deleted = 0`);
    organizations = arrOrg.map((item) => item.id);
    // return sendResponse(res, 400, "organizations is required");
  }
  // console.log(organizations);
  organizations = typeof organizations == "string" ? JSON.parse(organizations) : organizations;
  const parentIllnessTypeQuery = `
  SELECT *
  FROM illness_type
  WHERE 
      ${organizations.map(() => `JSON_CONTAINS(organization, CAST(? AS JSON), '$')`).join(" AND ")}
      AND deleted = 0;
`;

  const params = [...organizations];
  // console.log(parentIllnessTypeQuery, "parentIllnessTypeQuery")
  const [parentIllnessTypeList] = await db.query(parentIllnessTypeQuery, params);
  // const uniqueObj = {};
  // parentIllnessTypeList.forEach((item) => {
  //   if (!uniqueObj[item.name]) {
  //     uniqueObj[item.name] = item;
  //   }
  // });
  // const uniqueArr = Object.values(uniqueObj);
  // console.log(uniqueArr);
  // return uniqueArr;

  return sendResponse(res, 200, parentIllnessTypeList);
};

/** Function to view all or Illness Type */
export const viewIllnessTypeDropdown = async (req, res) => {
  let fetchParentQuery = `
      SELECT id, name, parent, organization 
      FROM illness_type 
      WHERE deleted = 0 AND parent IS NULL
    `;
  let [parentRecords] = await db.query(fetchParentQuery);

  if (parentRecords.length === 0) {
    return sendResponse(res, 200, []);
  }

  const fetchChildrenRecursively = async (parentId) => {
    let fetchChildQuery = `
        SELECT id, name, parent, organization 
        FROM illness_type 
        WHERE deleted = 0 AND parent = ?
      `;
    const [children] = await db.query(fetchChildQuery, [parentId]);

    for (const child of children) {
      const grandchildren = await fetchChildrenRecursively(child.id);
      children.push(...grandchildren);
    }
    return children;
  };

  for (const parent of parentRecords) {
    const children = await fetchChildrenRecursively(parent.id);
    parentRecords.push(...children);
  }
  parentRecords = await decodeAndParseFields(parentRecords);
  return sendResponse(res, 200, parentRecords);
};
// export const viewIllnessTypeDropdown = async (req, res) => {
//   const { id } = req.params;
//   const condition = await whereCondition({
//     table: "illness_type",
//     page: req.query.page,
//     all: req.query.all,
//     pageSize: req.query.pageSize,
//     filter: req.query.filter,
//     id,
//     user: req.user,
//     grouped: req.query.grouped,
//   });

//   const searchFields = ["illness_type.name", "illness_type.description", "users.name", "organization.name"];
//   let searchCondition = await searchConditionRecord(req.query.search, searchFields);
//   const joins = [
//     {
//       type: "left",
//       targetTable: "users",
//       onCondition: "users.id = illness_type.created_by",
//     },
//     {
//       type: "left",
//       targetTable: "organization",
//       onCondition: "organization.id = illness_type.organization",
//     },
//   ];
//   const joinsRecord = await makeJoins(joins);

//   const fetchQuery = `SELECT illness_type.*, users.name AS created_by, organization.name AS organization_name FROM illness_type ${joinsRecord} WHERE illness_type.deleted = 0 ${searchCondition} ${condition}`;
//   let [records] = await db.query(fetchQuery);

//   for (const record of records) {
//     record.description = await decodeSingle_statement(record.description);
//     if (req.query.grouped == "true") {
//       const [organizations] = await db.query(`SELECT organization FROM illness_type WHERE deleted = 0 AND name = ?`, [
//         record.name,
//       ]);
//       const arr = organizations.map((item) => item.organization);
//       record.organizations = arr;
//     }
//   }

//   const totalRecord = await countQueryCondition(fetchQuery);
//   return sendResponse(res, 200, records, totalRecord);
// };
