import db, { buildSelectQuery } from "../db-config.js";
import { getCalendar } from "../helper/calendar.js";
import { uploadToDDRM } from "../helper/ddrmUploader.js";
import {
  checkCircularDependency,
  countQueryCondition,
  createQueryBuilder,
  decodeAndParseFields,
  deleteRecord,
  encodeAndStringifyFields,
  getOrganizationAccordingToDepartment,
  insertActivityLog,
  makeJoins,
  searchConditionRecord,
  uniqueIdGenerator,
  updateQueryBuilder,
  uploadFile,
  whereCondition,
} from "../helper/general.js";
import { sendResponse } from "../helper/wrapper.js";
import ActionTemplateList from "../sequelize/ActionTemplateListSchema.js";
import CustomActionCreation from "../sequelize/CustomActionCreationSchema.js";

function sortTasks(tasks) {
  const graph = new Map();
  const reverseGraph = new Map();

  // Build the graph and reverse graph
  tasks.forEach((task) => {
    graph.set(task.id, new Set());
    reverseGraph.set(task.id, new Set());
  });

  tasks.forEach((task) => {
    const successors = Array.isArray(task.successor_task_template)
      ? task.successor_task_template
      : [task.successor_task_template].filter(Boolean);
    const predecessors = Array.isArray(task.predecessor_task_template)
      ? task.predecessor_task_template
      : [task.predecessor_task_template].filter(Boolean);

    successors.forEach((succId) => {
      if (graph.has(succId)) {
        graph.get(task.id).add(succId);
        reverseGraph.get(succId).add(task.id);
      }
    });

    predecessors.forEach((predId) => {
      if (graph.has(predId)) {
        graph.get(predId).add(task.id);
        reverseGraph.get(task.id).add(predId);
      }
    });
  });

  function dfs(node, visited, stack, graph) {
    visited.add(node);
    for (let neighbor of graph.get(node)) {
      if (!visited.has(neighbor)) {
        dfs(neighbor, visited, stack, graph);
      }
    }
    stack.push(node);
  }

  function topologicalSort(graph) {
    const visited = new Set();
    const stack = [];
    for (let node of graph.keys()) {
      if (!visited.has(node)) {
        dfs(node, visited, stack, graph);
      }
    }
    return stack.reverse();
  }

  // Perform topological sort on both graphs
  const forwardOrder = topologicalSort(graph);
  const backwardOrder = topologicalSort(reverseGraph);

  // Combine both orders
  const scoreMap = new Map();
  forwardOrder.forEach((id, index) => {
    scoreMap.set(id, index);
  });
  backwardOrder.forEach((id, index) => {
    scoreMap.set(id, (scoreMap.get(id) || 0) + index);
  });

  // Final sorting based on combined scores
  const sortedIds = [...scoreMap.entries()].sort((a, b) => a[1] - b[1]).map((entry) => entry[0]);

  // Map the sorted IDs back to the original task objects
  const taskMap = new Map(tasks.map((task) => [task.id, task]));
  return sortedIds.map((id) => taskMap.get(id));
}

async function findValidTasks(taskId, selectedType, searchType, allTasks) {
  const validTasks = [];
  const task = allTasks.find((t) => t.id === taskId);

  if (!task) {
    return validTasks;
  }

  for (const potentialTask of allTasks) {
    if (potentialTask.id === taskId) continue;

    let isValid = true;

    if (selectedType === "successor" && searchType === "predecessor") {
      // Check if adding potentialTask as predecessor would create a cycle
      isValid = !(await checkCycleInChain(potentialTask.id, taskId, allTasks));
    } else if (selectedType === "predecessor" && searchType === "successor") {
      // Check if adding potentialTask as successor would create a cycle
      isValid = !(await checkCycleInChain(taskId, potentialTask.id, allTasks));
    }

    if (isValid) {
      validTasks.push(potentialTask);
    }
  }

  return validTasks;
}

async function checkCycleInChain(start, end, allTasks) {
  const visited = new Set();

  function dfs(currentId) {
    if (currentId === end) return true;
    if (visited.has(currentId)) return false;

    visited.add(currentId);

    const currentTask = allTasks.find((t) => t.id === currentId);
    if (currentTask) {
      if (currentTask.successor_task_template && dfs(currentTask.successor_task_template)) {
        return true;
      }
    }

    return false;
  }

  return dfs(start);
}

function fetchLinkedTasks(actionTemplateList, selectedTaskId) {
  const taskMap = new Map();

  actionTemplateList.forEach((task) => {
    taskMap.set(task.id, task);
  });

  const result = new Set();

  function dfs(taskId) {
    if (!taskMap.has(taskId) || result.has(taskId)) return;

    const task = taskMap.get(taskId);
    result.add(taskId); // Add the task to the result set

    if (task.successor_task_template) {
      dfs(task.successor_task_template);
    }

    if (task.predecessor_task_template) {
      dfs(task.predecessor_task_template);
    }
  }

  dfs(selectedTaskId);

  return Array.from(result).map((id) => taskMap.get(id));
}

export const getValidTasks = async (req, res) => {
  const { task_id, selected_type, organization } = req.query;

  if (!selected_type || (selected_type !== "successor" && selected_type !== "predecessor")) {
    return sendResponse(res, 400, "Invalid selected_type. Must be 'successor' or 'predecessor'.");
  }

  try {
    const [allTasks] = await db.query(
      `SELECT id, predecessor_task_template, successor_task_template 
       FROM action_template_list 
       WHERE organization = ?`,
      [organization]
    );

    const searchType = selected_type === "successor" ? "predecessor" : "successor";
    const validTasks = await findValidTasks(task_id, selected_type, searchType, allTasks);

    if (validTasks.length === 0) {
      return sendResponse(res, 200, "No valid tasks found", []);
    }

    const validTaskIds = validTasks.map((task) => task.id);
    const [fullValidTasks] = await db.query(`SELECT * FROM action_template_list WHERE id IN (?)`, [validTaskIds]);

    return sendResponse(res, 200, "Valid tasks retrieved successfully", fullValidTasks);
  } catch (error) {
    console.error("Error in getValidTasks:", error);
    return sendResponse(res, 500, "An error occurred while retrieving valid tasks");
  }
};

/** Function to create and update Custom Action Creation */
export const createUpdateCustomActionCreation = async (req, res) => {
  let { id, task_data, department, sidebar_id } = req.body;
  // console.log(req.body);
  if (req.body.template === "undefined") {
    delete req.body.template;
  }

  if (department) {
    req.body.organization = (await getOrganizationAccordingToDepartment(department))[0].organization;
  }
  if (task_data.predecessor_task_template || task_data.successor_task_template) {
    const circularDependency = await checkCircularDependency(
      id, // This might be undefined for new records
      task_data.predecessor_task_template,
      task_data.successor_task_template,
      "custom_action_creation"
    );
    if (circularDependency) {
      return sendResponse(res, 400, "Circular dependency detected. Cannot create or update the record.");
    }
  }
  let status = id ? "Updated" : "Created";

  req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;
  let attachment = [];

  // console.log('task_data: ', JSON.parse(task_data));

  task_data = JSON.parse(task_data);

  // for (let i = 0; req.body[`task_data[${i}][ddrm_id]`] != undefined; i++) {
  //   {
  //     const file = req.body[`task_data[${i}][ddrm_id]`];
  //     console.log('file: ', file);
  //     if (file) {
  //       // attachment.push(await uploadFile("custom_task_creation", file));
  //       // const ddrm_id = await uploadToDDRM(sidebar_id, file, req);
  //       const ddrm_id = file;
  //       attachment.push(ddrm_id);
  //     } else {
  //       attachment.push(req.body[`task_data[${i}][ddrm_id]`]);
  //     }
  //   }
  // }
  for (let i = 0; task_data?.[i]?.["ddrm_id"] != undefined; i++) {
    {
      // const file = req.body[`task_data[${i}][ddrm_id]`];
      const file = task_data[i]["ddrm_id"];
      // console.log('file: ', file);
      if (file) {
        // attachment.push(await uploadFile("custom_task_creation", file));
        // const ddrm_id = await uploadToDDRM(sidebar_id, file, req);
        const ddrm_id = file;
        attachment.push(ddrm_id);
      } else {
        // attachment.push(req.body[`task_data[${i}][ddrm_id]`]);
        attachment.push(task_data[i]["ddrm_id"]);
      }
    }
  }

  if (id != undefined) {
    const [result] = await db.query(`SELECT * FROM custom_action_creation WHERE id = ${id}`);
    await db.query(
      `UPDATE custom_action_creation SET predecessor_task_template = ${result[0]?.predecessor_task_template} WHERE id = ${result[0]?.successor_task_template}`
    );
    await db.query(
      `UPDATE custom_action_creation SET successor_task_template = ${result[0]?.successor_task_template} WHERE id = ${result[0]?.predecessor_task_template}`
    );
  }

  // console.log("-->>>>>>",task_data)

  const arr = [];
  let i = 0;
  for (let action of task_data) {
    action.id = req.body.id;
    action.template = req.body.template;
    action.organization = req.body.organization;
    action.department = req.body.department;
    action[id ? "updated_by" : "created_by"] = req.user.sessionid;
    action.ddrm_id = attachment[i] ? attachment[i++] : null;
    console.log("module", action.module);
    const startDate = new Date(action.start_date);
    const endDate = new Date(action.end_date);

    const differenceInMilliseconds = endDate - startDate;

    const differenceInDays = differenceInMilliseconds / (1000 * 60 * 60 * 24);

    const totalWorkingHours = differenceInDays * 8;
    action.estimated_time = totalWorkingHours;
    action = await encodeAndStringifyFields(action);

    if (!id) {
      const unique_id = await uniqueIdGenerator(
        req.body.organization,
        department,
        "CustomActionCreation",
        "custom_action_creation",
        "unique_id",
        "unique_id"
      );
      action.unique_id = unique_id;
    }

    const { query, values } = id
      ? req.body.save_type == "template"
        ? updateQueryBuilder(ActionTemplateList, action)
        : updateQueryBuilder(CustomActionCreation, action)
      : req.body.save_type == "template"
        ? createQueryBuilder(ActionTemplateList, action)
        : createQueryBuilder(CustomActionCreation, action);
    // console.log(query);
    const [result] = await db.query(query, values);
    if (action.predecessor_task_template) {
      let predId = action.predecessor_task_template;
      await db.query(
        `UPDATE custom_action_creation SET successor_task_template = ${id ? id : result.insertId} WHERE id = ${predId}`
      );
    }
    if (action.successor_task_template) {
      let sucId = action.successor_task_template;
      await db.query(
        `UPDATE custom_action_creation SET predecessor_task_template = ${id ? id : result.insertId} WHERE id = ${sucId}`
      );
    }
    arr.push({ name: action?.name, id: id ? id : result.insertId });
  }

  /** Insert record for activity log */
  await insertActivityLog(
    req.user.sessionid,
    id ? "update" : "create",
    "Custom Action Creation",
    `This user ${req.user.sessionid} has ${status} a new custom action`
  );

  return sendResponse(res, 200, `Record ${status} successfully`);
};

/** Function to get Custom Action Creation list */
export const getCustomActionCreation = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "custom_action_creation",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });
  const searchTableName = [
    "custom_action_creation.task_title",
    "custom_action_creation.task_description",
    "custom_action_creation.start_date",
    "custom_action_creation.end_date",
    // "custom_action_creation.priority_level",
    "custom_action_creation.status",
    "custom_action_creation.task_budget",
    "createdUser.name",
    // "custom_action_creation.task_resources",
    "organization.name",
    "department.name",
  ];
  const searchCondition = await searchConditionRecord(req.query.search, searchTableName);

  const joins = [
    {
      type: "left",
      targetTable: "users as createdUser",
      onCondition: "custom_action_creation.created_by = createdUser.id",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "custom_action_creation.organization = organization.id",
    },
    {
      type: "left",
      targetTable: "users as u2",
      onCondition: "u2.id = custom_action_creation.updated_by",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "custom_action_creation.department = department.id",
    },
    {
      type: "left",
      targetTable: "users as u1",
      onCondition: "u1.id = custom_action_creation.responsible_person",
    },
    {
      type: "left",
      targetTable: "roles",
      onCondition: "custom_action_creation.role = roles.id",
    },
    {
      type: "left",
      targetTable: "skills",
      onCondition: "custom_action_creation.skill = skills.id",
    },
    {
      type: "left",
      targetTable: "labels",
      onCondition: "custom_action_creation.label = labels.id",
    },
    {
      type: "left",
      targetTable: "certificates",
      onCondition: "custom_action_creation.certificate = certificates.id",
    },
    {
      type: "left",
      targetTable: "action_template_list",
      onCondition: "custom_action_creation.template = action_template_list.id",
    },
    {
      type: "left",
      targetTable: "category",
      onCondition: "custom_action_creation.tag = category.id",
    },
    {
      type: "left",
      targetTable: "custom_action_creation AS suc",
      onCondition: "custom_action_creation.successor_task_template = suc.id",
    },
    {
      type: "left",
      targetTable: "custom_action_creation AS pred",
      onCondition: "custom_action_creation.predecessor_task_template = pred.id",
    },
    {
      type: "left",
      targetTable: "sidebar AS moduleTable",
      onCondition: "moduleTable.id = custom_action_creation.module",
    },
    {
      type: "left",
      targetTable: "sidebar AS subModuleTable",
      onCondition: "subModuleTable.id = custom_action_creation.sub_module",
    },
    {
      type: "left",
      targetTable: "repository",
      onCondition: "repository.id = custom_action_creation.ddrm_id",
    },
  ];

  const joinCondition = await makeJoins(joins);

  // Extract the date and the ordering/pagination parts
  let dynamicDate = '';
  let orderLimitOffset = '';
  const dateMatch = condition.match(/custom_action_creation.start_date\s*=\s*'(\d{4}-\d{2}-\d{2})'/);
  if (dateMatch && dateMatch[1]) {
    dynamicDate = dateMatch[1];
    // Extract everything after the date condition
    const dateIndex = condition.indexOf(dateMatch[0]);
    orderLimitOffset = condition.substring(dateIndex + dateMatch[0].length).trim();
  }

  // Create new date range condition
  let dateCondition = '';
  if (dynamicDate) {
    dateCondition = `AND custom_action_creation.start_date <= '${dynamicDate}' AND custom_action_creation.end_date >= '${dynamicDate}'`;
  } else {
    dateCondition = '';
    // If no date found, use the entire original condition minus the AND
    orderLimitOffset = condition.replace(/^AND\s*/, '').trim();
  }

  // Build full condition by appending the original ordering/pagination
  const fullCondition = `${dateCondition} ${orderLimitOffset}`;

  const customActionCreationQuery = `SELECT custom_action_creation.* , action_template_list.name AS template_name    , createdUser.name AS created_by_name , createdUser.surname AS created_by_surname, createdUser.profile AS created_by_profile , organization.name AS organization_name , department.name AS department_name , CONCAT(u1.name , ' ' , u1.surname) AS responsible_person_name , u1.profile AS responsible_person_profile , CONCAT(u2.name , ' ' ,u2.surname ) AS updated_by_name , u2.profile AS updated_by_profile , roles.name AS role_name , skills.name AS skill_name , labels.name AS label_name , certificates.name AS certificate_name , category.name AS tag_name , pred.task_title AS predecessor_task_template_name , suc.task_title AS successor_task_template_name , moduleTable.title AS module_name , subModuleTable.title AS sub_module_name , 
  repository.url as file FROM custom_action_creation ${joinCondition} WHERE custom_action_creation.deleted = 0 AND custom_action_creation.approved = 1 ${searchCondition}  ${fullCondition}`;

  // console.log("customActionCreationQuery ", customActionCreationQuery);

  let [customActionCreation] = await db.query(customActionCreationQuery);
  customActionCreation = await decodeAndParseFields(customActionCreation);
  
  for (let element of customActionCreation) {
    const ids = element?.collaborators || [];
    if (element.human_resource_requirements) {
      for (const resource of element.human_resource_requirements) {
        const [roleN] = await db.query(`SELECT name FROM roles WHERE id = '${resource.role}'`);
        resource.role_name = roleN[0]?.name;
      }
    }

    let arr = [];

    if (ids) {
      for (const id of ids) {
        const [result] = await db.query(
          `SELECT CONCAT(name , ' ', surname) AS name , id , profile FROM users WHERE id = '${id}'`
        );
        arr.push({
          name: result[0]?.name,
          id: result[0]?.id,
          profile: result[0]?.profile,
        });
      }
    }
    element.collaborator_details = arr;
    element.file = element.file ? element.file : "";
  }

  const totalRecord = await countQueryCondition(customActionCreationQuery);
  if (req.query.type == `dependency`) customActionCreation = sortTasks(customActionCreation);

  return sendResponse(res, 200, customActionCreation, totalRecord);
};

export const deleteCustomActionCreation = async (req, res) => {
  const { id } = req.params;
  await deleteRecord(CustomActionCreation, id);
  await insertActivityLog(req.user.sessionid, "delete", "Custom Action Creation", id);
  return sendResponse(res, 200, "Record deleted successfully");
};

export const fetchLinkedTemplates = async (req, res) => {
  let { organization, template } = req.query;
  let [result] = await db.query(`SELECT * FROM action_template_list WHERE organization = '${organization}'`);
  result = await decodeAndParseFields(result);
  for (let r of result) {
    r.human_resource_requirements = [{ role: "", no_of_peoples: "" }];
    r.technological_resources = [""];
  }
  template = Number(template);

  if (req.query.type == "true") {
    result = fetchLinkedTasks(result, template);
  } else {
    if (template) {
      [result] = await db.query(`SELECT * FROM action_template_list WHERE id = '${template}'`);
    }
    result = await decodeAndParseFields(result);
    for (let r of result) {
      r.human_resource_requirements = [{ role: "", no_of_peoples: "" }];
      r.technological_resources = [""];
    }
  }

  return sendResponse(res, 200, result);
};

export const fetchGanttChart = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "custom_action_creation",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
    filter: req.query.filter,
  });
  const searchTableName = [
    "custom_action_creation.task_title",
    "custom_action_creation.task_description",
    "custom_action_creation.start_date",
    "custom_action_creation.end_date",
    "custom_action_creation.priority_level",
    "custom_action_creation.status",
    "custom_action_creation.task_budget",
    "custom_action_creation.task_resources",
    "organization.name",
    "department.name",
  ];
  const searchCondition = await searchConditionRecord(req.query.search, searchTableName);

  const joins = [
    {
      type: "left",
      targetTable: "users as createdUser",
      onCondition: "custom_action_creation.created_by = createdUser.id",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "custom_action_creation.organization = organization.id",
    },
    {
      type: "left",
      targetTable: "users as u2",
      onCondition: "u2.id = custom_action_creation.updated_by",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "custom_action_creation.department = department.id",
    },
    {
      type: "left",
      targetTable: "users as u1",
      onCondition: "u1.id = custom_action_creation.responsible_person",
    },
  ];

  const joinCondition = await makeJoins(joins);

  const customActionCreationQuery = `SELECT custom_action_creation.* , createdUser.name AS created_by_name , createdUser.surname AS created_by_surname, createdUser.profile AS created_by_profile , organization.name AS organization_name , department.name AS department_name , CONCAT(u1.name , ' ' , u1.surname) AS assigned_to_name , u1.profile AS assigned_to_profile , CONCAT(u2.name , ' ' ,u2.surname ) AS updated_by_name , u2.profile AS updated_by_profile FROM custom_action_creation ${joinCondition} WHERE custom_action_creation.deleted = 0 AND custom_action_creation.approved = 1  ${searchCondition}  ${condition}`;
  let [customActionCreation] = await db.query(customActionCreationQuery);
  for (let i = 0; i < customActionCreation.length; i++) {
    const element = customActionCreation[i];
    element.start = element?.start_date;
    element.end = element?.end_date;
    element.type = "task";
    element.dependencies = element?.predecessor_task_template ? [element?.predecessor_task_template] : [];
    element.progress = 25;
    element.name = element?.task_title;
  }
  customActionCreation = await decodeAndParseFields(customActionCreation);
  customActionCreation = sortTasks(customActionCreation);
  // customActionCreation = customActionCreation.reverse();
  const totalRecord = await countQueryCondition(customActionCreationQuery);
  return sendResponse(res, 200, customActionCreation, totalRecord);
};

export const getCustomActionCalender = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "custom_action_creation",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });
  const searchTableName = [
    "custom_action_creation.task_title",
    "custom_action_creation.task_description",
    "custom_action_creation.start_date",
    "custom_action_creation.end_date",
    "custom_action_creation.priority_level",
    "custom_action_creation.status",
    "custom_action_creation.task_budget",
    "custom_action_creation.task_resources",
    "organization.name",
    "department.name",
  ];
  const searchCondition = await searchConditionRecord(req.query.search, searchTableName);

  const joins = [
    {
      type: "left",
      targetTable: "users as createdUser",
      onCondition: "custom_action_creation.created_by = createdUser.id",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "custom_action_creation.organization = organization.id",
    },
    {
      type: "left",
      targetTable: "users as u2",
      onCondition: "u2.id = custom_action_creation.updated_by",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "custom_action_creation.department = department.id",
    },
    {
      type: "left",
      targetTable: "users as u1",
      onCondition: "u1.id = custom_action_creation.responsible_person",
    },
  ];

  const joinCondition = await makeJoins(joins);

  const customActionCreationQuery = `SELECT custom_action_creation.*  ,  createdUser.name AS created_by_name , createdUser.surname AS created_by_surname, createdUser.profile AS created_by_profile , organization.name AS organization_name , department.name AS department_name , CONCAT(u1.name , ' ' , u1.name) AS assigned_to_name , u1.profile AS assigned_to_profile , CONCAT(u2.name , ' ' ,u2.surname ) AS updated_by_name , u2.profile AS updated_by_profile FROM custom_action_creation ${joinCondition} WHERE custom_action_creation.deleted = 0 ${searchCondition}  ${condition}`;
  let [customActionCreation] = await db.query(customActionCreationQuery);
  customActionCreation = await decodeAndParseFields(customActionCreation);
  const templateStructure = {
    start: "start_date",
    end: "end_date",
    summary: "task_title",
  };
  const calenderPath = await getCalendar(customActionCreation, templateStructure, "custom_action_creation");
  const totalRecord = await countQueryCondition(customActionCreationQuery);
  return sendResponse(res, 200, calenderPath, totalRecord);
};
export const fetchEmployeeAccordingToGivenData = async (req, res) => {
  const { role, skill } = req.query;
  if (skill && role) {
    let [result] = await db.query("SELECT * FROM users WHERE role = ? AND JSON_CONTAINS(skills, ?)", [role, skill]);
    result = await decodeAndParseFields(result);
    for (const user of result) {
      const id = user.id;
      const [total] = await db.query(
        `SELECT COUNT(*) AS total FROM custom_action_creation WHERE responsible_person = '${id}' AND deleted = '0'`
      );
      user.task = total[0]?.total ? total[0]?.total : 0;
    }
    return sendResponse(res, 200, result);
  } else {
    return sendResponse(res, 200, []);
  }
};

export const kanbanStatusUpdate = async (req, res) => {
  const { id, status } = req.body;
  await db.query(`UPDATE custom_action_creation SET status = '${status}' WHERE id = '${id}'`);

  return sendResponse(res, 200, "Record updated successfully");
};

export const getKanbanData = async (req, res) => {
  req.query.filter = JSON.parse(req.query.filter);
  let { organization, date } = req.query.filter;

  const data = [
    {
      id: "1",
      status: "To-Do",
      title: "🏢 To Do",
      tasks: [],
    },
    {
      id: "2",
      status: "In-Progress",
      title: "👨‍🏭 In Progress",
      tasks: [],
    },
    {
      id: "3",
      status: "Completed",
      title: "✅ Completed",
      tasks: [],
    },
  ];

  let condition = "";
  for (const ele of Object.entries(req.query.filter)) {
    if (ele[1] != "" && ele[1] != null) {
      // ele[1] = Number(ele[1]);
      if (ele[0] === "created_at") {
        ele[0] = `DATE(${ele[0]})`;
      }
      condition += ` AND ${ele[0]} = '${ele[1]}'`;
    }
  }
  console.log("condition", condition);
  let [result] = await db.query(`SELECT * FROM custom_action_creation WHERE deleted = 0 ${condition}`);
  // console.log(req.query);
  // console.log(organization);
  result = await decodeAndParseFields(result);

  for (const element of result) {
    let arr = [];
    const collaborators = element.collaborators;
    if (collaborators) {
      // console.log(collaborators);
      for (const collaborator of collaborators) {
        const [user] = await db.query(
          `SELECT id , CONCAT(name , ' ' , surname) AS name , profile FROM users WHERE id = '${collaborator}' AND deleted = '0'`
        );
        // console.log(user);
        arr.push({
          id: user[0]?.id,
          name: user[0]?.name,
          profile: user[0]?.profile,
        });
      }
    }
    // console.log(arr);
    const index = data.findIndex((d) => d.status === element.status);
    data[index]?.tasks.push(element);
    element.collaborator_details = arr;
  }
  return sendResponse(res, 200, data);
};

export const createUpdateActionFunction = async (data, req) => {
  let { id, task_data, department, sidebar_id } = data;
  // console.log(data);
  if (department) {
    data.organization = (await getOrganizationAccordingToDepartment(department))[0].organization;
  }
  if (task_data.predecessor_task_template || task_data.successor_task_template) {
    const circularDependency = await checkCircularDependency(
      id, // This might be undefined for new records
      task_data.predecessor_task_template,
      task_data.successor_task_template,
      "custom_action_creation"
    );
    if (circularDependency) {
      throw new Error("Circular dependency detected. Cannot create or update the record.");
    }
  }
  let status = task_data[0]?.id ? "Updated" : "Created";

  data[task_data[0]?.id ? "updated_by" : "created_by"] = data.user.sessionid;
  let attachment = [];
  if (data.files && data.files[`task_data[0][file]`] != undefined) {
    for (let i = 0; data.files[`task_data[${i}][file]`] != undefined; i++) {
      {
        const file = data.files[`task_data[${i}][file]`];
        if (file) {
          // attachment.push(await uploadFile("custom_task_creation", file));
          const ddrm_id = await uploadToDDRM(sidebar_id, file, req);
          attachment.push(ddrm_id);
        } else {
          console.log(data[`task_data[${i}][file]`], "data[`task_data[${i}][file]`]");
          attachment.push(data[`task_data[${i}][ddrm_id]`]);
        }
      }
    }
  } else {
    let actions = task_data;
    if (typeof task_data == "string") {
      actions = JSON.parse(task_data);
    }
    for (let i = 0; i < actions.length; i++) {
      {
        const action = actions[i];
        if (action.ddrm_id) {
          attachment.push(action.ddrm_id);
        }
      }
    }
  }

  if (id != undefined) {
    const [result] = await db.query(`SELECT * FROM custom_action_creation WHERE id = ${id}`);
    await db.query(
      `UPDATE custom_action_creation SET predecessor_task_template = ${result[0]?.predecessor_task_template} WHERE id = ${result[0]?.successor_task_template}`
    );
    console.log("result[0]?.successor_task_template: ", result[0]?.successor_task_template);
    await db.query(
      `UPDATE custom_action_creation SET successor_task_template = ${result[0]?.successor_task_template} WHERE id = ${result[0]?.predecessor_task_template}`
    );
  }

  if (typeof task_data == "string") {
    task_data = JSON.parse(task_data);
  }

  const arr = [];
  let i = 0;
  for (let action of task_data) {
    delete action.updated_at;
    action.template = data.template;
    action.organization = data.organization;
    action.department = data.department;
    action[action.id ? "updated_by" : "created_by"] = data.user.sessionid;
    action.ddrm_id = attachment[i] ? attachment[i++] : null;
    const startDate = new Date(action.start_date);
    const endDate = new Date(action.end_date);

    const differenceInMilliseconds = endDate - startDate;

    const differenceInDays = differenceInMilliseconds / (1000 * 60 * 60 * 24);

    const totalWorkingHours = differenceInDays * 8;
    action.estimated_time = totalWorkingHours;
    action = await encodeAndStringifyFields(action);

    if (!id) {
      const unique_id = await uniqueIdGenerator(
        req.body.organization,
        department,
        "CustomActionCreation",
        "custom_action_creation",
        "unique_id",
        "unique_id"
      );
      action.unique_id = unique_id;
    }
    const { query, values } = action.id
      ? data.save_type == "template"
        ? updateQueryBuilder(ActionTemplateList, action)
        : updateQueryBuilder(CustomActionCreation, action)
      : data.save_type == "template"
        ? createQueryBuilder(ActionTemplateList, action)
        : createQueryBuilder(CustomActionCreation, action);
    // console.log(query);
    const [result] = await db.query(query, values);
    if (action.predecessor_task_template) {
      let predId = action.predecessor_task_template;
      await db.query(
        `UPDATE custom_action_creation SET successor_task_template = ${id ? id : result.insertId} WHERE id = ${predId}`
      );
    }
    if (action.successor_task_template) {
      let sucId = action.successor_task_template;
      await db.query(
        `UPDATE custom_action_creation SET predecessor_task_template = ${id ? id : result.insertId} WHERE id = ${sucId}`
      );
    }
    arr.push({
      name: action?.name,
      id: action.id ? action.id : result.insertId,
    });
  }

  /** Insert record for activity log */
  await insertActivityLog(
    data.user.sessionid,
    id ? "update" : "create",
    "Custom Action Creation",
    `This user ${data.user.sessionid} has ${status} a new custom action`
  );
  return { status: true, arr };
};

/** Function to get Custom Action Creation list */
export const getCustomActionFunction = async (req) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "custom_action_creation",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });
  const searchTableName = [
    "custom_action_creation.task_title",
    "custom_action_creation.task_description",
    "custom_action_creation.start_date",
    "custom_action_creation.end_date",
    "custom_action_creation.priority_level",
    "custom_action_creation.status",
    "custom_action_creation.task_budget",
    "custom_action_creation.task_resources",
    "organization.name",
    "department.name",
  ];
  const searchCondition = await searchConditionRecord(req.query.search, searchTableName);

  const joins = [
    {
      type: "left",
      targetTable: "users as createdUser",
      onCondition: "custom_action_creation.created_by = createdUser.id",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "custom_action_creation.organization = organization.id",
    },
    {
      type: "left",
      targetTable: "users as u2",
      onCondition: "u2.id = custom_action_creation.updated_by",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "custom_action_creation.department = department.id",
    },
    {
      type: "left",
      targetTable: "users as u1",
      onCondition: "u1.id = custom_action_creation.responsible_person",
    },
    {
      type: "left",
      targetTable: "roles",
      onCondition: "custom_action_creation.role = roles.id",
    },
    {
      type: "left",
      targetTable: "skills",
      onCondition: "custom_action_creation.skill = skills.id",
    },
    {
      type: "left",
      targetTable: "labels",
      onCondition: "custom_action_creation.label = labels.id",
    },
    {
      type: "left",
      targetTable: "certificates",
      onCondition: "custom_action_creation.certificate = certificates.id",
    },
    {
      type: "left",
      targetTable: "action_template_list",
      onCondition: "custom_action_creation.template = action_template_list.id",
    },
    {
      type: "left",
      targetTable: "category",
      onCondition: "custom_action_creation.tag = category.id",
    },
    {
      type: "left",
      targetTable: "custom_action_creation AS suc",
      onCondition: "custom_action_creation.successor_task_template = suc.id",
    },
    {
      type: "left",
      targetTable: "custom_action_creation AS pred",
      onCondition: "custom_action_creation.predecessor_task_template = pred.id",
    },
    {
      type: "left",
      targetTable: "sidebar AS moduleTable",
      onCondition: "moduleTable.id = custom_action_creation.module",
    },
    {
      type: "left",
      targetTable: "sidebar AS subModuleTable",
      onCondition: "subModuleTable.id = custom_action_creation.sub_module",
    },
    {
      type: "left",
      targetTable: "repository",
      onCondition: "repository.id = custom_action_creation.ddrm_id",
    },
  ];

  const joinCondition = await makeJoins(joins);

  const customActionCreationQuery = `SELECT custom_action_creation.* , action_template_list.name AS template_name    , createdUser.name AS created_by_name , createdUser.surname AS created_by_surname, createdUser.profile AS created_by_profile , organization.name AS organization_name , department.name AS department_name , CONCAT(u1.name , ' ' , u1.surname) AS responsible_person_name , u1.profile AS responsible_person_profile , CONCAT(u2.name , ' ' ,u2.surname ) AS updated_by_name , u2.profile AS updated_by_profile , roles.name AS role_name , skills.name AS skill_name , labels.name AS label_name , certificates.name AS certificate_name , category.name AS tag_name , pred.task_title AS predecessor_task_template_name , suc.task_title AS successor_task_template_name , moduleTable.title AS module_name , subModuleTable.title AS sub_module_name, repository.url AS attachment FROM custom_action_creation ${joinCondition} WHERE custom_action_creation.deleted = 0  ${searchCondition}  ${condition}`;
  // console.log(customActionCreationQuery);
  let [customActionCreation] = await db.query(customActionCreationQuery);
  customActionCreation = await decodeAndParseFields(customActionCreation);
  for (const element of customActionCreation) {
    // if record_name is a number (id of other table)
    if (!isNaN(element.record_name) && element.record_name !== null && element.record_name !== "") {
      const parsedValue = Number(element.record_name);
      const [tableName] = await db.query(`SELECT table_name FROM sidebar WHERE id = ?`, [element.sub_module]);
      if (tableName[0]?.table_name) {
        // const mappedFields = await buildSelectQuery(tableName[0].table_name);
        // let [tableResult] = await db.query(
        //   `SELECT ${mappedFields} FROM ${tableName[0].table_name} WHERE organization = ? AND id = ? AND deleted = 0 ORDER BY id DESC`,
        //   [element.organization, parsedValue]
        // );
        // element.record = tableResult[0]?.name;

        if (tableName[0].table_name === "ppe_repository") {
          const [ppe] = await db.query(
            `SELECT ppe_type.name FROM ppe_repository LEFT JOIN ppe_type ON ppe_repository.type = ppe_type.id WHERE ppe_repository.organization = ? AND ppe_repository.id = ? AND ppe_repository.deleted = 0 ORDER BY ppe_repository.id DESC`,
            [element.organization, parsedValue]
          );
          element.record = ppe[0]?.name;
        } else {
          const mappedFields = await buildSelectQuery(tableName[0].table_name);
          let [tableResult] = await db.query(
            `SELECT ${mappedFields} FROM ${tableName[0].table_name} WHERE organization = ? AND id = ? AND deleted = 0 ORDER BY id DESC`,
            [element.organization, parsedValue]
          );
          element.record = tableResult[0]?.name;
        }
      }
    }
    const ids = element.collaborators || [];
    if (element.human_resource_requirements) {
      for (const resource of element.human_resource_requirements) {
        const [roleN] = await db.query(`SELECT name FROM roles WHERE id = '${resource.role}'`);
        resource.role_name = roleN[0]?.name;
      }
    }
    let arr = [];
    for (const id of ids) {
      const [result] = await db.query(
        `SELECT CONCAT(name , ' ', surname) AS name , id , profile FROM users WHERE id = '${id}'`
      );
      arr.push({
        name: result[0]?.name,
        id: result[0]?.id,
        profile: result[0]?.profile,
      });
    }

    element.collaborator_details = arr;
  }
  const totalRecord = await countQueryCondition(customActionCreationQuery);
  if (req.query.type == `dependency`) customActionCreation = sortTasks(customActionCreation);

  return customActionCreation;
};

export const getStructuredCustomActionCreation = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "custom_action_creation",
    page: req.query.page,
    all: req.query.all,
    id,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    user: req.user,
  });

  const joins = [
    {
      type: "left",
      targetTable: "users as createdUser",
      onCondition: "custom_action_creation.created_by = createdUser.id",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "custom_action_creation.organization = organization.id",
    },
    {
      type: "left",
      targetTable: "users as u2",
      onCondition: "u2.id = custom_action_creation.updated_by",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "custom_action_creation.department = department.id",
    },
    {
      type: "left",
      targetTable: "users as u1",
      onCondition: "u1.id = custom_action_creation.responsible_person",
    },
    {
      type: "left",
      targetTable: "roles",
      onCondition: "custom_action_creation.role = roles.id",
    },
    {
      type: "left",
      targetTable: "skills",
      onCondition: "custom_action_creation.skill = skills.id",
    },
    {
      type: "left",
      targetTable: "labels",
      onCondition: "custom_action_creation.label = labels.id",
    },
    {
      type: "left",
      targetTable: "certificates",
      onCondition: "custom_action_creation.certificate = certificates.id",
    },
    {
      type: "left",
      targetTable: "action_template_list",
      onCondition: "custom_action_creation.template = action_template_list.id",
    },
    {
      type: "left",
      targetTable: "category",
      onCondition: "custom_action_creation.tag = category.id",
    },
    {
      type: "left",
      targetTable: "custom_action_creation AS suc",
      onCondition: "custom_action_creation.successor_task_template = suc.id",
    },
    {
      type: "left",
      targetTable: "custom_action_creation AS pred",
      onCondition: "custom_action_creation.predecessor_task_template = pred.id",
    },
    {
      type: "left",
      targetTable: "sidebar AS moduleTable",
      onCondition: "moduleTable.id = custom_action_creation.module",
    },
    {
      type: "left",
      targetTable: "sidebar AS subModuleTable",
      onCondition: "subModuleTable.id = custom_action_creation.sub_module",
    },
  ];
  const joinCondition = await makeJoins(joins);

  // Fetch data with necessary joins
  const customActionQuery = `SELECT custom_action_creation.*, 
      moduleTable.title AS module_name, moduleTable.id AS module,
      subModuleTable.title AS sub_module_name, subModuleTable.id AS sub_module 
    FROM custom_action_creation 
    ${joinCondition} 
    WHERE custom_action_creation.deleted = 0 ${condition}`;

  let [customActions] = await db.query(customActionQuery);
  customActions = await decodeAndParseFields(customActions);

  // Organize data into the desired structure
  const structuredData = customActions.reduce((acc, action) => {
    const moduleIndex = acc.findIndex((mod) => mod.module === action.module);

    if (moduleIndex === -1) {
      // Add a new module with sub_module and action
      acc.push({
        module: action.module,
        module_name: action.module_name,
        sub_modules: [
          {
            sub_module: action.sub_module,
            sub_module_name: action.sub_module_name,
            records: [action],
          },
        ],
      });
    } else {
      const subModuleIndex = acc[moduleIndex].sub_modules.findIndex(
        (subMod) => subMod.sub_module === action.sub_module
      );

      if (subModuleIndex === -1) {
        // Add a new sub_module with the action
        acc[moduleIndex].sub_modules.push({
          sub_module: action.sub_module,
          sub_module_name: action.sub_module_name,
          records: [action],
        });
      } else {
        // Add action to existing sub_module's records
        acc[moduleIndex].sub_modules[subModuleIndex].records.push(action);
      }
    }

    return acc;
  }, []);
  console.log(structuredData);
  const totalRecord = await countQueryCondition(customActionQuery);
  return sendResponse(res, 200, structuredData, totalRecord);
};
