import asyncHandler from "express-async-handler";
import db from "../db-config.js";
import {
  checkNameInOrganization,
  countQueryCondition,
  dateValidator,
  deleteFile,
  deleteRecord,
  getOrganizationAccordingToDepartment,
  insertActivityLog,
  makeJoins,
  whereCondition,
  uniqueIdGenerator,
  uploadFile,
  updateQueryBuilder,
  createQueryBuilder,
  searchConditionRecord,
  decodeAndParseFields,
  encodeAndStringifyFields,
  processUploadDocuments,
} from "../helper/general.js";
import ContractorPermitLicense from "../sequelize/ContractorPermitLicenseSchema.js";
import { sendResponse } from "../helper/wrapper.js";

/**Function to create and update Contractor Permit License  */
export const createUpdatedContractorPermitLicense = async (req, res) => {
  const { id, title, department } = req.body;

  if (department) {
    req.body.organization = (
      await getOrganizationAccordingToDepartment(department)
    )[0].organization;
  }

  /**Unique id generator */
  const permitLicenseId = await uniqueIdGenerator(
    req.body.organization,
    department,
    "CPL",
    "contractor_permit_license",
    "permit_license_id"
  );

  let result;
  if (id) {
    /**Update query for permit license */
    req.body.updated_by = req.user.sessionid;
    req.body = await encodeAndStringifyFields(req.body);
    const { query, values } = updateQueryBuilder(
      ContractorPermitLicense,
      req.body
    );
    [result] = await db.query(query, values);

    /**Insert record for activity log */
    await insertActivityLog(
      req.user.sessionid,
      "update",
      "Contractor Permit License",
      `This user Update Contractor Permit License which id is ${id}`
    );

    return sendResponse(res, 200, "Record updated successfully");
  } else {
    req.body.created_by = req.user.sessionid;
    req.body.permit_license_id = permitLicenseId;
    req.body = await encodeAndStringifyFields(req.body);

    const unique_id = await uniqueIdGenerator(
      req.body.organization ,
      department,
      "ContractorPermitLicense",
      "contractor_permit_license",
      "unique_id",
      "unique_id"
    );
    req.body.unique_id = unique_id;

    const { query, values } = createQueryBuilder(
      ContractorPermitLicense,
      req.body
    );
    [result] = await db.query(query, values);

    /** Insert record for activity log */
    await insertActivityLog(
      req.user.sessionid,
      "create",
      "Contractor Permit License",
      `This user created a new Contractor Permit License named '${title}' for organization ${checkNameInOrganization.organization}`
    );

    return sendResponse(res, 200, "Record created successfully");
  }
};

/**Function to view all Contractor Permit License */
export const viewAllContractorPermitLicense = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "contractor_permit_license",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });

  const searchTableName = [
    "contractor_permit_license.title",
    "contractor_permit_license.description",
    "contractor_permit_license.permit_type",
    "CONCAT(employee.name , ' ' , employee.surname)",
    "organization.name",
    "issuing_authority.name",
    "CONCAT(created_by.name , ' ' , created_by.surname)",
  ];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = contractor_permit_license.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = contractor_permit_license.organization",
    },
    {
      type: "left",
      targetTable: "issuing_authority",
      onCondition:
        "issuing_authority.id = contractor_permit_license.issuing_authority",
    },
    {
      type: "left",
      targetTable: "permit_license",
      onCondition: "permit_license.id = contractor_permit_license.title",
    },
    {
      type: "left",
      targetTable: "contractor_registration",
      onCondition:
        "contractor_registration.id = contractor_permit_license.permit_holder",
    },
    {
      type: "left",
      targetTable: "contractor_registration AS cr",
      onCondition: "cr.id = contractor_permit_license.contractor",
    },
    {
      type: "left",
      targetTable: "project_registration",
      onCondition:
        "project_registration.id =  contractor_permit_license.project ",
    },
    {
      type: "left",
      targetTable: "users as employee",
      onCondition: "employee.id =  contractor_permit_license.employee",
    },
    {
      type: "left",
      targetTable: "roles as employee_role",
      onCondition: "employee_role.id =  employee.role",
    },
    {
      type: "left",
      targetTable: "department as employee_department",
      onCondition: "employee_department.id =  employee.department",
    },
    {
      type: "left",
      targetTable: "users as created_by",
      onCondition: "created_by.id =  contractor_permit_license.created_by",
    },
  ];
  const joinsRecord = await makeJoins(joins);

  /**Record of all alert */
  const permitLicenseDataFetchQuery = `SELECT 
    contractor_permit_license.* ,
    contractor_permit_license.created_by as created_by_id, 
    CONCAT(created_by.name , ' ' , created_by.surname) as created_by_name,
    cr.contractor_name AS contractor_name, 
    cr.cipc_registration_number AS contractor_cipc_registration_number,
    project_registration.project_title  AS project_name,
    contractor_registration.contractor_name AS permit_holder_name,
    organization.name as organization_name, 
    issuing_authority.name as issuing_authority_name , 
    CONCAT(employee.name , ' ' , employee.surname) as employee_name,
    employee.unique_id as employee_number,
    employee_role.name as role_name,
    employee_department.name as employee_department,
    permit_license.name as permit_license_name,
    permit_license.description as permit_license_description
    FROM contractor_permit_license  
    ${joinsRecord} 
    where contractor_permit_license.deleted = 0 ${searchCondition} ${condition}`;
  // return console.log(permitLicenseDataFetchQuery)
  let [permitLicenseDataFetch] = await db.query(permitLicenseDataFetchQuery);

  permitLicenseDataFetch = await decodeAndParseFields(permitLicenseDataFetch);
  /**Count all Contractor Permit License */

  const totalRecord = await countQueryCondition(permitLicenseDataFetchQuery);
  return sendResponse(res, 200, permitLicenseDataFetch, totalRecord);
};

/**Function to delete a specific Contractor Permit License */
export const deleteContractorPermitLicense = async (req, res) => {
  const { id } = req.params;
  const deleteRecordQuery = await deleteRecord("contractor_permit_license", id);
  if (deleteRecordQuery) {
    /**Insert record for activity log */
    insertActivityLog(
      req.user.sessionid,
      "delete",
      "Contractor Permit License",
      id
    );
    return sendResponse(
      res,
      200,
      "Permit and License Record Deleted successfully"
    );
  }
};
