import db from "../db-config.js";
import {
  countQueryCondition,
  createQueryBuilder,
  decodeAndParseFields,
  encodeAndStringifyFields,
  insertActivityLog,
  makeJoins,
  searchConditionRecord,
  updateQueryBuilder,
  whereCondition,
} from "../helper/general.js";
import { sendResponse } from "../helper/wrapper.js";
import ActionActivities from "../sequelize/ActionActivitiesSchema.js";

export const createUpdateActivity = async (req, res) => {
  let { custom_action_id, type } = req.body;
  custom_action_id = Number(custom_action_id);
  let status = req.body.id ? "Updated" : "Created";
  req.body[req.body.id ? "updated_by" : "created_by"] = req.user.sessionid;

  // Function to check predecessors recursively
  async function checkPredecessors(taskId, checkedTasks = new Set()) {
    if (checkedTasks.has(taskId)) {
      return { completed: true }; // Already checked this task, avoid infinite loop
    }
    checkedTasks.add(taskId);

    const [tasks] = await db.query(
      "SELECT id, status, predecessor_task_template FROM custom_action_creation WHERE id = ?",
      [taskId]
    );

    if (tasks.length === 0) {
      return { completed: true };
    }

    const task = tasks[0];

    if (task.status !== "Completed") {
      return { completed: false, incompleteTaskId: task.id };
    }

    if (task.predecessor_task_template) {
      const predecessorIds = JSON.parse(task.predecessor_task_template);
      // console.log("pred");
      // console.log(predecessorIds);
      const result = await checkPredecessors(predecessorIds, checkedTasks);
      if (!result.completed) {
        return result; // Propagate the incomplete task ID up the chain
      }
    }

    return { completed: true };
  }

  // Check if the custom action exists
  const [orgFetch] = await db.query(
    "SELECT * FROM custom_action_creation WHERE id = ?",
    [custom_action_id]
  );

  if (orgFetch.length === 0) {
    return sendResponse(res, 404, "Custom action not found");
  }

  // Perform deep check of predecessors
  if (req.body.type == "start") {
    const checkResult = await checkPredecessors(
      orgFetch[0].predecessor_task_template
    );
    if (!checkResult.completed) {
      return sendResponse(
        res,
        400,
        `Cannot proceed. Predecessor task with ID ${checkResult.incompleteTaskId} is not completed.`
      );
    }
  }
  // The rest of your existing logic
  req.body.action =
    type == "start"
      ? `Started the timer for this task`
      : "Ended the timer for this task";

  const startTime = orgFetch[0]?.start_date;
  const endTime = orgFetch[0]?.end_date;
  // console.log(orgFetch);
  const startDate = new Date(startTime);
  const endDate = new Date(endTime);

  const differenceInMilliseconds = endDate - startDate;
  const differenceInDays = differenceInMilliseconds / (1000 * 60 * 60 * 24);
  const totalWorkingHours = differenceInDays * 8;

  req.body.estimated_time = totalWorkingHours;
  req.body.organization = orgFetch[0]?.organization;
  req.body.department = orgFetch[0]?.department;
  if (req.body.status) {
    await db.query(
      `UPDATE custom_action_creation SET status =  '${req.body.status}' WHERE id = ${custom_action_id}`
    );
    req.body.type = req.body.status;
    req.body.action = `Changed the status of this task to ${req.body.status}`;
    const { query, values } = req.body.id
      ? updateQueryBuilder(ActionActivities, req.body)
      : createQueryBuilder(ActionActivities, req.body);
    await db.query(query, values);
    return sendResponse(res, 200, `Record ${status} Successfully`);
  }

  req.body = await encodeAndStringifyFields(req.body);

  const { query, values } = req.body.id
    ? updateQueryBuilder(ActionActivities, req.body)
    : createQueryBuilder(ActionActivities, req.body);

  await db.query(query, values);
  await db.query(
    `UPDATE custom_action_creation SET type = '${type}' WHERE id = ?`,
    [custom_action_id]
  );
  await insertActivityLog(
    req.user.sessionid,
    status,
    "Action Activities",
    req.body.id
  );
  return sendResponse(res, 200, `Record ${status} Successfully`);
};

const getWorkTimeChartData = async () => {
  const [results] = await db.query(`
    SELECT 
      DAYOFWEEK(created_at) AS weekday_number,
      DAYNAME(created_at) AS day,
      SUM(estimated_time) AS total_time
    FROM custom_action_creation
    WHERE deleted = 0
    GROUP BY DAYOFWEEK(created_at), DAYNAME(created_at)
    ORDER BY weekday_number
  `);
  // Initialize all 6 days
  const daysOrder = [
    "Monday",
    "Tuesday",
    "Wednesday",
    "Thursday",
    "Friday",
    "Saturday",
  ];
  const workDataMap = {};

  for (let day of daysOrder) {
    workDataMap[day] = 0;
  }

  // Fill values from DB result
  results.forEach((row) => {
    const { day, total_time } = row;
    if (workDataMap.hasOwnProperty(day)) {
      workDataMap[day] = parseFloat(total_time);
    }
  });

  // Prepare final output
  const data = daysOrder.map((day) => workDataMap[day]);
  const categories = daysOrder.map((day) => [day]);

  return {
    data,
    categories,
  };
};

export const getActionActivity = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "action_activities",
    id,
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    user: req.user,
    filterTable: "custom_action_creation",
  });

  const searchTableName = [
    "action_activities.type",
    "CONCAT(users.name ,  ' '  , users.surname)",
    "action_activities.description",
    "organization.name",
    "department.name",
  ];

  const searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = action_activities.created_by",
    },
    {
      type: "left",
      targetTable: "custom_action_creation",
      onCondition:
        "custom_action_creation.id = action_activities.custom_action_id",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = users.my_organization",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = custom_action_creation.department",
    },
  ];

  const joinCondition = await makeJoins(joins);

  const activityQuery = `SELECT action_activities.* , custom_action_creation.task_title AS task_name ,custom_action_creation.unique_id,  organization.name AS organization_name , department.name AS department_name ,  CONCAT(users.name ,  ' '  , users.surname) AS created_by_name , users.profile AS created_by_profile FROM action_activities ${joinCondition} WHERE action_activities.deleted = 0 ${searchCondition} ${condition}`;
  // console.log('activityQuery: ', activityQuery);

  const [result] = await db.query(
    `SELECT * FROM custom_action_creation WHERE deleted = 0`
  );
  const chartData = [
    { name: "To-Do", data: 0 },
    { name: "In-Progress", data: 0 },
    { name: "Completed", data: 0 },
    { name: "Over-Due", data: 0 },
    { name: "Pending", data: 0 },
    // { name: "Approved", data: 0 },
    { name: "Total", data: 0 },
  ];
  let totalData = 0;
  for (const item of result) {
    const index = chartData.findIndex((obj) => obj.name === item.status);
    if (index >= 0) {
      chartData[index].data = chartData[index].data + 1;
      totalData = totalData + 1;
    }
  }

  chartData[chartData.length - 1].data = totalData;
  let [activity] = await db.query(activityQuery);

  activity = await decodeAndParseFields(activity);

  const total = await countQueryCondition(activityQuery);

  const workTimeChartData = await getWorkTimeChartData()

  return res.status(200).json({
    status: true,
    data: activity,
    total,
    tasks: chartData,
    workTimeChartData,
  });
};
